"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAddBulkToTimelineAction = void 0;
var _react = require("react");
var _reactRedux = require("react-redux");
var _public = require("@kbn/data-plugin/public");
var _securitysolutionDataTable = require("@kbn/securitysolution-data-table");
var _kuery = require("../../../../common/lib/kuery");
var _kibana = require("../../../../common/lib/kibana");
var _constants = require("../../../../../common/constants");
var _sourcerer = require("../../../../common/containers/sourcerer");
var _containers = require("../../../../timelines/containers");
var _selectors = require("../../../../common/components/events_viewer/selectors");
var _use_update_timeline = require("../../../../timelines/components/open_timeline/use_update_timeline");
var _store = require("../../../../timelines/store");
var _use_create_timeline = require("../../../../timelines/hooks/use_create_timeline");
var _translations = require("../translations");
var _timeline = require("../../../../../common/types/timeline");
var _timeline2 = require("../../../../../common/api/timeline");
var _actions = require("../actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  setEventsLoading,
  setSelected
} = _securitysolutionDataTable.dataTableActions;
/*
 * useAddBulkToTimelineAction  returns a bulk action that can be passed to the
 * TGrid so that multiple items at a time can be added to the timeline.
 *
 * It also syncs the timerange passed to the TGrid to that of  timeline
 *
 * */
const useAddBulkToTimelineAction = ({
  localFilters,
  tableId,
  from,
  to,
  scopeId
}) => {
  const [disableActionOnSelectAll, setDisabledActionOnSelectAll] = (0, _react.useState)(false);
  const {
    browserFields,
    dataViewId,
    runtimeMappings,
    indexPattern,
    // important to get selectedPatterns from useSourcererDataView
    // in order to include the exclude filters in the search that are not stored in the timeline
    selectedPatterns
  } = (0, _sourcerer.useSourcererDataView)(scopeId);
  const dispatch = (0, _reactRedux.useDispatch)();
  const {
    uiSettings
  } = (0, _kibana.useKibana)().services;
  const {
    filters,
    dataTable: {
      selectAll,
      totalCount,
      sort,
      selectedEventIds
    } = _securitysolutionDataTable.tableDefaults
  } = (0, _reactRedux.useSelector)(state => (0, _selectors.eventsViewerSelector)(state, tableId));
  const esQueryConfig = (0, _react.useMemo)(() => (0, _public.getEsQueryConfig)(uiSettings), [uiSettings]);
  const timelineQuerySortField = (0, _react.useMemo)(() => {
    return sort.map(({
      columnId,
      columnType,
      esTypes,
      sortDirection
    }) => ({
      field: columnId,
      direction: sortDirection,
      esTypes: esTypes !== null && esTypes !== void 0 ? esTypes : [],
      type: columnType
    }));
  }, [sort]);
  const combinedFilters = (0, _react.useMemo)(() => [...localFilters, ...filters], [localFilters, filters]);
  const combinedQuery = (0, _react.useMemo)(() => {
    return (0, _kuery.combineQueries)({
      config: esQueryConfig,
      dataProviders: [],
      indexPattern,
      filters: combinedFilters,
      kqlQuery: {
        query: '',
        language: 'kuery'
      },
      browserFields,
      kqlMode: 'filter'
    });
  }, [esQueryConfig, indexPattern, combinedFilters, browserFields]);
  const filterQuery = (0, _react.useMemo)(() => {
    if (!combinedQuery) return '';
    return combinedQuery.filterQuery;
  }, [combinedQuery]);

  // in case user selects all items, we will use below handler to get the IDs of
  // all items (with max limit)
  const [,, searchhandler] = (0, _containers.useTimelineEventsHandler)({
    dataViewId,
    endDate: to,
    startDate: from,
    id: tableId,
    fields: ['_id', 'timestamp'],
    sort: timelineQuerySortField,
    indexNames: selectedPatterns,
    filterQuery,
    runtimeMappings,
    limit: Math.min(_constants.BULK_ADD_TO_TIMELINE_LIMIT, totalCount),
    timerangeKind: 'absolute'
  });
  (0, _react.useEffect)(() => {
    if (!selectAll) {
      setDisabledActionOnSelectAll(false);
      return;
    }
    if (totalCount > _constants.BULK_ADD_TO_TIMELINE_LIMIT) {
      setDisabledActionOnSelectAll(true);
    } else {
      setDisabledActionOnSelectAll(false);
    }
  }, [selectAll, totalCount]);
  const clearActiveTimeline = (0, _use_create_timeline.useCreateTimeline)({
    timelineId: _timeline.TimelineId.active,
    timelineType: _timeline2.TimelineType.default
  });
  const updateTimelineIsLoading = (0, _react.useCallback)(payload => dispatch(_store.timelineActions.updateIsLoading(payload)), [dispatch]);
  const updateTimeline = (0, _use_update_timeline.useUpdateTimeline)();
  const createTimeline = (0, _react.useCallback)(async ({
    timeline,
    ruleNote,
    timeline: {
      filters: eventIdFilters
    }
  }) => {
    var _timeline$indexNames;
    await clearActiveTimeline();
    updateTimelineIsLoading({
      id: _timeline.TimelineId.active,
      isLoading: false
    });
    updateTimeline({
      duplicate: true,
      from,
      id: _timeline.TimelineId.active,
      notes: [],
      timeline: {
        ...timeline,
        indexNames: (_timeline$indexNames = timeline.indexNames) !== null && _timeline$indexNames !== void 0 ? _timeline$indexNames : [],
        show: true,
        filters: eventIdFilters
      },
      to,
      ruleNote
    });
  }, [updateTimeline, updateTimelineIsLoading, clearActiveTimeline, from, to]);
  const sendBulkEventsToTimelineHandler = (0, _react.useCallback)(items => {
    (0, _actions.sendBulkEventsToTimelineAction)(createTimeline, items.map(item => item.ecs), 'KqlFilter');
    dispatch(setSelected({
      id: tableId,
      isSelectAllChecked: false,
      isSelected: false,
      eventIds: selectedEventIds
    }));
  }, [dispatch, createTimeline, selectedEventIds, tableId]);
  const onActionClick = (0, _react.useCallback)((items, isAllSelected, setLoading, clearSelection) => {
    if (!items) return;
    /*
     * Trigger actions table passed isAllSelected param
     *
     * and selectAll is used when using DataTable
     * */
    const onResponseHandler = localResponse => {
      sendBulkEventsToTimelineHandler(localResponse.events);
      if (tableId === _securitysolutionDataTable.TableId.alertsOnAlertsPage) {
        setLoading(false);
        clearSelection();
      } else {
        dispatch(setEventsLoading({
          id: tableId,
          isLoading: false,
          eventIds: Object.keys(selectedEventIds)
        }));
      }
    };
    if (isAllSelected || selectAll) {
      if (tableId === _securitysolutionDataTable.TableId.alertsOnAlertsPage) {
        setLoading(true);
      } else {
        dispatch(setEventsLoading({
          id: tableId,
          isLoading: true,
          eventIds: Object.keys(selectedEventIds)
        }));
      }
      searchhandler(onResponseHandler);
      return;
    }
    sendBulkEventsToTimelineHandler(items);
    clearSelection();
  }, [dispatch, selectedEventIds, tableId, searchhandler, selectAll, sendBulkEventsToTimelineHandler]);
  const investigateInTimelineTitle = (0, _react.useMemo)(() => {
    return disableActionOnSelectAll ? `${_translations.INVESTIGATE_BULK_IN_TIMELINE} ( max ${_constants.BULK_ADD_TO_TIMELINE_LIMIT} )` : _translations.INVESTIGATE_BULK_IN_TIMELINE;
  }, [disableActionOnSelectAll]);
  const memoized = (0, _react.useMemo)(() => ({
    label: investigateInTimelineTitle,
    key: 'add-bulk-to-timeline',
    'data-test-subj': 'investigate-bulk-in-timeline',
    disableOnQuery: disableActionOnSelectAll,
    onClick: onActionClick
  }), [disableActionOnSelectAll, investigateInTimelineTitle, onActionClick]);
  return memoized;
};
exports.useAddBulkToTimelineAction = useAddBulkToTimelineAction;