"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RelatedAlertsByProcessAncestry = void 0;
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _helpers = require("../../../../helpers");
var _use_action_cell_data_provider = require("../table/use_action_cell_data_provider");
var _use_alert_prevalence_from_process_tree = require("../../../containers/alerts/use_alert_prevalence_from_process_tree");
var _insight_accordion = require("./insight_accordion");
var _simple_alert_table = require("./simple_alert_table");
var _investigate_in_timeline_button = require("../table/investigate_in_timeline_button");
var _translations = require("../../../../detections/components/alerts_table/translations");
var _translations2 = require("./translations");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dataProviderLimit = 5;

/**
 * Fetches and displays alerts that were generated in the associated process'
 * process tree.
 * Offers the ability to dive deeper into the investigation by opening
 * the related alerts in a timeline investigation.
 *
 * In contrast to other insight accordions, this one does not fetch the
 * count and alerts on mount since the call to fetch the process tree
 * and its associated alerts is quite expensive.
 * The component requires users to click on the accordion in order to
 * initiate the fetch of the associated events.
 *
 * In order to achieve this, this component orchestrates two helper
 * components:
 *
 * RelatedAlertsByProcessAncestry (empty cache)
 *     user clicks -->
 *         FetchAndNotifyCachedAlertsByProcessAncestry (fetches data, shows loading state)
 *     cache loaded -->
 *         ActualRelatedAlertsByProcessAncestry (displays data)
 *
 * The top-level component maintains a "cache" state that is used for
 * state management and to prevent double-fetching in case the
 * accordion is closed and re-opened.
 *
 * Due to the ephemeral nature of the data, it was decided to keep the
 * state inside the component rather than to add it to Redux.
 */
const RelatedAlertsByProcessAncestry = exports.RelatedAlertsByProcessAncestry = /*#__PURE__*/_react.default.memo(({
  originalDocumentId,
  index,
  eventId,
  scopeId
}) => {
  const [showContent, setShowContent] = (0, _react.useState)(false);
  const [cache, setCache] = (0, _react.useState)({});
  const onToggle = (0, _react.useCallback)(isOpen => setShowContent(isOpen), []);

  // Makes sure the component is not fetching data before the accordion
  // has been openend.
  const renderContent = (0, _react.useCallback)(() => {
    if (!showContent) {
      return null;
    } else if (cache.alertIds) {
      return /*#__PURE__*/_react.default.createElement(ActualRelatedAlertsByProcessAncestry, {
        eventId: eventId,
        scopeId: scopeId,
        alertIds: cache.alertIds
      });
    }
    return /*#__PURE__*/_react.default.createElement(FetchAndNotifyCachedAlertsByProcessAncestry, {
      index: index,
      originalDocumentId: originalDocumentId,
      eventId: eventId,
      isActiveTimelines: (0, _helpers.isActiveTimeline)(scopeId !== null && scopeId !== void 0 ? scopeId : ''),
      onCacheLoad: setCache
    });
  }, [showContent, cache.alertIds, index, originalDocumentId, eventId, scopeId]);
  return /*#__PURE__*/_react.default.createElement(_insight_accordion.InsightAccordion, {
    prefix: "RelatedAlertsByProcessAncestry"
    // `renderContent` and the associated sub-components are making sure to
    // render the correct loading and error states so we can omit these states here
    ,
    state: "success",
    text:
    // If we have fetched the alerts, display the count here, otherwise omit the count
    cache.alertIds ? (0, _translations2.PROCESS_ANCESTRY_COUNT)(cache.alertIds.length) : _translations2.PROCESS_ANCESTRY,
    renderContent: renderContent,
    onToggle: onToggle
  });
});
RelatedAlertsByProcessAncestry.displayName = 'RelatedAlertsByProcessAncestry';

/**
 * Fetches data, displays a loading and error state and notifies about on success
 */
const FetchAndNotifyCachedAlertsByProcessAncestry = ({
  originalDocumentId,
  index,
  isActiveTimelines,
  onCacheLoad,
  eventId
}) => {
  const {
    values: indices
  } = index;
  const {
    values: wrappedDocumentId
  } = originalDocumentId;
  const documentId = Array.isArray(wrappedDocumentId) ? wrappedDocumentId[0] : '';
  const {
    loading,
    error,
    alertIds
  } = (0, _use_alert_prevalence_from_process_tree.useAlertPrevalenceFromProcessTree)({
    isActiveTimeline: isActiveTimelines,
    documentId,
    indices: indices !== null && indices !== void 0 ? indices : []
  });
  (0, _react.useEffect)(() => {
    if (alertIds && alertIds.length !== 0) {
      onCacheLoad({
        alertIds
      });
    }
  }, [alertIds, onCacheLoad]);
  if (loading) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, null);
  } else if (error) {
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, _translations2.PROCESS_ANCESTRY_ERROR);
  } else if (!alertIds || alertIds.length === 0) {
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, _translations2.PROCESS_ANCESTRY_EMPTY);
  }
  return null;
};
FetchAndNotifyCachedAlertsByProcessAncestry.displayName = 'FetchAndNotifyCachedAlertsByProcessAncestry';

/**
 * Renders the alert table and the timeline button from a filled cache.
 */
const ActualRelatedAlertsByProcessAncestry = ({
  alertIds,
  eventId,
  scopeId
}) => {
  const shouldUseFilters = alertIds && alertIds.length && alertIds.length >= dataProviderLimit;
  const dataProviders = (0, _react.useMemo)(() => {
    if (alertIds && alertIds.length) {
      if (shouldUseFilters) {
        return null;
      } else {
        return alertIds.reduce((result, alertId, index) => {
          const id = `${scopeId}-${eventId}-event.id-${index}-${alertId}`;
          result.push((0, _use_action_cell_data_provider.getDataProvider)('_id', id, alertId));
          return result;
        }, []);
      }
    }
    return null;
  }, [alertIds, shouldUseFilters, scopeId, eventId]);
  const filters = (0, _react.useMemo)(() => {
    if (shouldUseFilters) {
      return [{
        meta: {
          alias: _translations2.PROCESS_ANCESTRY_FILTER,
          type: 'phrases',
          key: '_id',
          params: [...alertIds],
          negate: false,
          disabled: false,
          value: alertIds.join()
        },
        query: {
          bool: {
            should: alertIds.map(id => {
              return {
                match_phrase: {
                  _id: id
                }
              };
            }),
            minimum_should_match: 1
          }
        }
      }];
    } else {
      return null;
    }
  }, [alertIds, shouldUseFilters]);
  if (!dataProviders && !filters) {
    return null;
  }
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_simple_alert_table.SimpleAlertTable, {
    alertIds: alertIds
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, null), /*#__PURE__*/_react.default.createElement(_investigate_in_timeline_button.InvestigateInTimelineButton, {
    asEmptyButton: false,
    dataProviders: dataProviders,
    filters: filters,
    "data-test-subj": "investigate-ancestry-in-timeline"
  }, _translations.ACTION_INVESTIGATE_IN_TIMELINE));
};
ActualRelatedAlertsByProcessAncestry.displayName = 'ActualRelatedAlertsByProcessAncestry';