"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isPolicySetToEventCollectionOnly = exports.ensureOnlyEventCollectionIsAllowed = exports.disableProtections = void 0;
var _lodash = require("lodash");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const allOsValues = [_types.PolicyOperatingSystem.mac, _types.PolicyOperatingSystem.linux, _types.PolicyOperatingSystem.windows];
const getPolicyProtectionsReference = () => [{
  keyPath: 'malware.mode',
  osList: [...allOsValues],
  disableValue: _types.ProtectionModes.off,
  enableValue: _types.ProtectionModes.prevent
}, {
  keyPath: 'ransomware.mode',
  osList: [_types.PolicyOperatingSystem.windows],
  disableValue: _types.ProtectionModes.off,
  enableValue: _types.ProtectionModes.prevent
}, {
  keyPath: 'memory_protection.mode',
  osList: [...allOsValues],
  disableValue: _types.ProtectionModes.off,
  enableValue: _types.ProtectionModes.prevent
}, {
  keyPath: 'behavior_protection.mode',
  osList: [...allOsValues],
  disableValue: _types.ProtectionModes.off,
  enableValue: _types.ProtectionModes.prevent
}, {
  keyPath: 'attack_surface_reduction.credential_hardening.enabled',
  osList: [_types.PolicyOperatingSystem.windows],
  disableValue: false,
  enableValue: true
}, {
  keyPath: 'antivirus_registration.enabled',
  osList: [_types.PolicyOperatingSystem.windows],
  disableValue: false,
  enableValue: true
}];

/**
 * Returns a copy of the passed `PolicyConfig` with all protections set to disabled.
 *
 * @param policy
 * @returns
 */
const disableProtections = policy => {
  const result = disableCommonProtections(policy);
  return {
    ...result,
    windows: {
      ...result.windows,
      ...getDisabledWindowsSpecificProtections(result),
      popup: {
        ...result.windows.popup,
        ...getDisabledWindowsSpecificPopups(result)
      }
    }
  };
};
exports.disableProtections = disableProtections;
const disableCommonProtections = policy => {
  return Object.keys(policy).reduce((acc, item) => {
    const os = item;
    if (!allOsValues.includes(os)) {
      return acc;
    }
    return {
      ...acc,
      [os]: {
        ...policy[os],
        ...getDisabledCommonProtectionsForOS(policy, os),
        popup: {
          ...policy[os].popup,
          ...getDisabledCommonPopupsForOS(policy, os)
        }
      }
    };
  }, policy);
};
const getDisabledCommonProtectionsForOS = (policy, os) => ({
  behavior_protection: {
    ...policy[os].behavior_protection,
    mode: _types.ProtectionModes.off
  },
  memory_protection: {
    ...policy[os].memory_protection,
    mode: _types.ProtectionModes.off
  },
  malware: {
    ...policy[os].malware,
    blocklist: false,
    on_write_scan: false,
    mode: _types.ProtectionModes.off
  }
});
const getDisabledCommonPopupsForOS = (policy, os) => ({
  behavior_protection: {
    ...policy[os].popup.behavior_protection,
    enabled: false
  },
  malware: {
    ...policy[os].popup.malware,
    enabled: false
  },
  memory_protection: {
    ...policy[os].popup.memory_protection,
    enabled: false
  }
});
const getDisabledWindowsSpecificProtections = policy => ({
  ransomware: {
    ...policy.windows.ransomware,
    mode: _types.ProtectionModes.off
  },
  attack_surface_reduction: {
    ...policy.windows.attack_surface_reduction,
    credential_hardening: {
      enabled: false
    }
  }
});
const getDisabledWindowsSpecificPopups = policy => ({
  ransomware: {
    ...policy.windows.popup.ransomware,
    enabled: false
  }
});

/**
 * Returns the provided with only event collection turned enabled
 * @param policy
 */
const ensureOnlyEventCollectionIsAllowed = policy => {
  const updatedPolicy = disableProtections(policy);
  (0, _lodash.set)(updatedPolicy, 'windows.antivirus_registration.mode', _types.AntivirusRegistrationModes.disabled);
  (0, _lodash.set)(updatedPolicy, 'windows.antivirus_registration.enabled', false);
  return updatedPolicy;
};

/**
 * Checks to see if the provided policy is set to Event Collection only
 */
exports.ensureOnlyEventCollectionIsAllowed = ensureOnlyEventCollectionIsAllowed;
const isPolicySetToEventCollectionOnly = policy => {
  const protectionsRef = getPolicyProtectionsReference();
  let message;
  const hasEnabledProtection = protectionsRef.some(({
    keyPath,
    osList,
    disableValue
  }) => {
    return osList.some(osValue => {
      const fullKeyPathForOs = `${osValue}.${keyPath}`;
      const currentValue = (0, _lodash.get)(policy, fullKeyPathForOs);
      const isEnabled = currentValue !== disableValue;
      if (isEnabled) {
        message = `property [${fullKeyPathForOs}] is set to [${currentValue}]`;
      }
      return isEnabled;
    });
  });
  return {
    isOnlyCollectingEvents: !hasEnabledProtection,
    message
  };
};
exports.isPolicySetToEventCollectionOnly = isPolicySetToEventCollectionOnly;