"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withLink = exports.useGetLinkUrl = exports.useGetLinkProps = exports.isSecurityId = exports.isModified = exports.getAppIdsFromId = exports.formatPath = exports.concatPaths = exports.LinkButton = exports.LinkAnchor = void 0;
var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _navigation = require("./navigation");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * It returns the `url` to use in link `href`.
 */
const useGetLinkUrl = () => {
  const {
    getAppUrl
  } = (0, _navigation.useGetAppUrl)();
  const getLinkUrl = (0, _react.useCallback)(({
    id,
    path: subPath = '',
    absolute = false,
    urlState
  }) => {
    const {
      appId,
      deepLinkId,
      path: mainPath = ''
    } = getAppIdsFromId(id);
    const path = concatPaths(mainPath, subPath);
    const formattedPath = urlState ? formatPath(path, urlState) : path;
    return getAppUrl({
      deepLinkId,
      appId,
      path: formattedPath,
      absolute
    });
  }, [getAppUrl]);
  return getLinkUrl;
};

/**
 * It returns the `onClick` and `href` props to use in link components based on the` deepLinkId` and `path` parameters.
 */
exports.useGetLinkUrl = useGetLinkUrl;
const useGetLinkProps = () => {
  const getLinkUrl = useGetLinkUrl();
  const {
    navigateTo
  } = (0, _navigation.useNavigateTo)();
  const getLinkProps = (0, _react.useCallback)(({
    id,
    path,
    urlState,
    onClick: onClickProps,
    overrideNavigation = true
  }) => {
    const url = getLinkUrl({
      id,
      path,
      urlState
    });
    return {
      href: url,
      onClick: ev => {
        if (isModified(ev)) {
          return;
        }
        if (onClickProps) {
          onClickProps(ev);
        }
        if (overrideNavigation) {
          ev.preventDefault();
          navigateTo({
            url
          });
        }
      }
    };
  }, [getLinkUrl, navigateTo]);
  return getLinkProps;
};

/**
 * HOC that wraps any Link component and makes it a navigation Link.
 */
exports.useGetLinkProps = useGetLinkProps;
const withLink = Component => /*#__PURE__*/_react.default.memo(function WithLink({
  id,
  path,
  urlState,
  onClick: _onClick,
  ...rest
}) {
  const getLink = useGetLinkProps();
  const {
    onClick,
    href
  } = getLink({
    id,
    path,
    urlState,
    onClick: _onClick,
    ...(rest.target === '_blank' && {
      overrideNavigation: false
    })
  });
  return /*#__PURE__*/_react.default.createElement(Component, (0, _extends2.default)({
    onClick: onClick,
    href: href
  }, rest));
});

/**
 * Security Solutions internal link button.
 *
 * `<LinkButton deepLinkId={SecurityPageName.hosts} />;`
 */
exports.withLink = withLink;
const LinkButton = exports.LinkButton = withLink(_eui.EuiButton);

/**
 * Security Solutions internal link anchor.
 *
 * `<LinkAnchor deepLinkId={SecurityPageName.hosts} />;`
 */
const LinkAnchor = exports.LinkAnchor = withLink(_eui.EuiLink);

// Utils

// External IDs are in the format `appId:deepLinkId` to match the Chrome NavLinks format.
// Internal Security Solution ids are the deepLinkId, the appId is omitted for convenience.
const isSecurityId = id => !id.includes(':');

// External links may contain an optional `path` in addition to the `appId` and `deepLinkId`.
// Format: `<appId>:<deepLinkId>/<path>`
exports.isSecurityId = isSecurityId;
const getAppIdsFromId = id => {
  const [linkId, strippedPath] = id.split(/\/(.*)/); // split by the first `/` character
  const path = strippedPath ? `/${strippedPath}` : '';
  if (!isSecurityId(linkId)) {
    const [appId, deepLinkId] = linkId.split(':');
    return {
      appId,
      deepLinkId,
      path
    };
  }
  return {
    deepLinkId: linkId,
    path
  }; // undefined `appId` for internal Security Solution links
};
exports.getAppIdsFromId = getAppIdsFromId;
const concatPaths = (path, subPath) => {
  if (path && subPath) {
    return `${path.replace(/\/$/, '')}/${subPath.replace(/^\//, '')}`;
  }
  return path || subPath || '';
};
exports.concatPaths = concatPaths;
const formatPath = (path, urlState) => {
  const urlStateClean = urlState.replace('?', '');
  const [urlPath, parameterPath] = path.split('?');
  let queryParams = '';
  if (urlStateClean && parameterPath) {
    queryParams = `?${parameterPath}&${urlStateClean}`;
  } else if (parameterPath) {
    queryParams = `?${parameterPath}`;
  } else if (urlStateClean) {
    queryParams = `?${urlStateClean}`;
  }
  return `${urlPath}${queryParams}`;
};
exports.formatPath = formatPath;
const isModified = event => event.metaKey || event.altKey || event.ctrlKey || event.shiftKey;
exports.isModified = isModified;