"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PrivilegeFormCalculator = void 0;
var _privilege_utils = require("../../../privilege_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calculator responsible for determining the displayed and effective privilege values for the following interfaces:
 * - <PrivilegeSpaceForm> and children
 * - <PrivilegeSpaceTable> and children
 */
class PrivilegeFormCalculator {
  constructor(kibanaPrivileges, role) {
    this.kibanaPrivileges = kibanaPrivileges;
    this.role = role;
  }

  /**
   * Returns the assigned base privilege.
   * If more than one base privilege is assigned, the most permissive privilege will be returned.
   * If no base privileges are assigned, then this will return `undefined`.
   *
   * @param privilegeIndex the index of the kibana privileges role component
   */
  getBasePrivilege(privilegeIndex) {
    const entry = this.role.kibana[privilegeIndex];
    const basePrivileges = this.kibanaPrivileges.getBasePrivileges(entry);
    return basePrivileges.find(bp => entry.base.includes(bp.id));
  }

  /**
   * Returns the ID of the *displayed* Primary Feature Privilege for the indicated feature and privilege index.
   * If the effective primary feature privilege is a "minimal" version, then this returns the corresponding non-minimal version.
   *
   * @example
   * The following kibana privilege entry will return `read`:
   * ```ts
   * const entry = {
   *    base: [],
   *    feature: {
   *       some_feature: ['minimal_read'],
   *    }
   * }
   * ```
   *
   * @param featureId the feature id to get the Primary Feature KibanaPrivilege for.
   * @param privilegeIndex the index of the kibana privileges role component
   * @param allSpacesSelected indicates if the privilege form is configured to grant access to all spaces.
   */
  getDisplayedPrimaryFeaturePrivilegeId(featureId, privilegeIndex, allSpacesSelected) {
    var _this$getDisplayedPri;
    return (_this$getDisplayedPri = this.getDisplayedPrimaryFeaturePrivilege(featureId, privilegeIndex, allSpacesSelected)) === null || _this$getDisplayedPri === void 0 ? void 0 : _this$getDisplayedPri.id;
  }

  /**
   * Determines if the indicated feature has sub-feature privilege assignments which differ from the "displayed" primary feature privilege.
   *
   * @param featureId the feature id
   * @param privilegeIndex the index of the kibana privileges role component
   * @param allSpacesSelected indicates if the privilege form is configured to grant access to all spaces.
   */
  hasCustomizedSubFeaturePrivileges(featureId, privilegeIndex, allSpacesSelected) {
    const feature = this.kibanaPrivileges.getSecuredFeature(featureId);
    const displayedPrimary = this.getDisplayedPrimaryFeaturePrivilege(featureId, privilegeIndex, allSpacesSelected);
    const formPrivileges = this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges([this.role.kibana[privilegeIndex]]);
    return feature.getSubFeaturePrivileges().some(sfp => {
      var _displayedPrimary$gra;
      const isGranted = formPrivileges.grantsPrivilege(sfp);
      const isGrantedByDisplayedPrimary = (_displayedPrimary$gra = displayedPrimary === null || displayedPrimary === void 0 ? void 0 : displayedPrimary.grantsPrivilege(sfp)) !== null && _displayedPrimary$gra !== void 0 ? _displayedPrimary$gra : isGranted;
      return isGranted !== isGrantedByDisplayedPrimary;
    });
  }

  /**
   * Returns the most permissive effective Primary Feature KibanaPrivilege, including the minimal versions.
   *
   * @param featureId the feature id
   * @param privilegeIndex the index of the kibana privileges role component
   * @param allSpacesSelected indicates if the privilege form is configured to grant access to all spaces.
   */
  getEffectivePrimaryFeaturePrivilege(featureId, privilegeIndex, allSpacesSelected) {
    const feature = this.kibanaPrivileges.getSecuredFeature(featureId);
    const basePrivilege = this.getBasePrivilege(privilegeIndex);
    const selectedFeaturePrivileges = this.getSelectedFeaturePrivileges(featureId, privilegeIndex);
    const effectivePrivilege = feature.getPrimaryFeaturePrivileges({
      includeMinimalFeaturePrivileges: true
    }).find(fp => {
      return selectedFeaturePrivileges.includes(fp.id) || (basePrivilege === null || basePrivilege === void 0 ? void 0 : basePrivilege.grantsPrivilege(fp));
    });
    const correctSpacesSelected = effectivePrivilege !== null && effectivePrivilege !== void 0 && effectivePrivilege.requireAllSpaces ? allSpacesSelected : true;
    const availablePrivileges = correctSpacesSelected && !(effectivePrivilege !== null && effectivePrivilege !== void 0 && effectivePrivilege.disabled);
    if (availablePrivileges) return effectivePrivilege;
  }

  /**
   * Determines if the indicated sub-feature privilege is granted.
   *
   * @param featureId the feature id
   * @param privilegeId the sub feature privilege id
   * @param privilegeIndex the index of the kibana privileges role component
   */
  isIndependentSubFeaturePrivilegeGranted(featureId, privilegeId, privilegeIndex) {
    const kibanaPrivilege = this.role.kibana[privilegeIndex];
    const feature = this.kibanaPrivileges.getSecuredFeature(featureId);
    const subFeaturePrivilege = feature.getSubFeaturePrivileges().find(ap => ap.id === privilegeId);
    const assignedPrivileges = this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges([kibanaPrivilege]);
    return assignedPrivileges.grantsPrivilege(subFeaturePrivilege);
  }

  /**
   * Returns the most permissive effective privilege within the indicated mutually-exclusive sub feature privilege group.
   *
   * @param featureId the feature id
   * @param subFeatureGroup the mutually-exclusive sub feature group
   * @param privilegeIndex the index of the kibana privileges role component
   */
  getSelectedMutuallyExclusiveSubFeaturePrivilege(featureId, subFeatureGroup, privilegeIndex) {
    const kibanaPrivilege = this.role.kibana[privilegeIndex];
    const assignedPrivileges = this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges([kibanaPrivilege]);
    return subFeatureGroup.privileges.find(p => {
      return assignedPrivileges.grantsPrivilege(p);
    });
  }

  /**
   * Determines if the indicated feature is capable of having its sub-feature privileges customized.
   *
   * @param featureId the feature id
   * @param privilegeIndex the index of the kibana privileges role component
   */
  canCustomizeSubFeaturePrivileges(featureId, privilegeIndex) {
    const selectedFeaturePrivileges = this.getSelectedFeaturePrivileges(featureId, privilegeIndex);
    const feature = this.kibanaPrivileges.getSecuredFeature(featureId);
    return feature.getPrimaryFeaturePrivileges({
      includeMinimalFeaturePrivileges: true
    }).some(apfp => selectedFeaturePrivileges.includes(apfp.id));
  }

  /**
   * Returns an updated set of feature privileges based on the toggling of the "Customize sub-feature privileges" control.
   *
   * @param featureId the feature id
   * @param privilegeIndex  the index of the kibana privileges role component
   * @param willBeCustomizing flag indicating if this feature is about to have its sub-feature privileges customized or not
   * @param allSpacesSelected indicates if the privilege form is configured to grant access to all spaces.
   */
  updateSelectedFeaturePrivilegesForCustomization(featureId, privilegeIndex, willBeCustomizing, allSpacesSelected) {
    const primary = this.getDisplayedPrimaryFeaturePrivilege(featureId, privilegeIndex, allSpacesSelected);
    const selectedFeaturePrivileges = this.getSelectedFeaturePrivileges(featureId, privilegeIndex);
    if (!primary) {
      return selectedFeaturePrivileges;
    }
    const nextPrivileges = [];
    if (willBeCustomizing) {
      const feature = this.kibanaPrivileges.getSecuredFeature(featureId);
      const startingPrivileges = feature.getSubFeaturePrivileges().filter(ap => primary.grantsPrivilege(ap)).map(p => p.id);
      nextPrivileges.push(primary.getMinimalPrivilegeId(), ...startingPrivileges);
    } else {
      nextPrivileges.push(primary.id);
    }
    return nextPrivileges;
  }

  /**
   * Determines if the indicated privilege entry is less permissive than the configured "global" entry for the role.
   * @param privilegeIndex the index of the kibana privileges role component
   */
  hasSupersededInheritedPrivileges(privilegeIndex) {
    const global = this.locateGlobalPrivilege(this.role);
    const entry = this.role.kibana[privilegeIndex];
    if ((0, _privilege_utils.isGlobalPrivilegeDefinition)(entry) || !global) {
      return false;
    }
    const globalPrivileges = this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges([global]);
    const formPrivileges = this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges([entry]);
    const hasAssignedBasePrivileges = this.kibanaPrivileges.getBasePrivileges(entry).some(base => entry.base.includes(base.id));
    const featuresWithDirectlyAssignedPrivileges = this.kibanaPrivileges.getSecuredFeatures().filter(feature => feature.getAllPrivileges().some(privilege => {
      var _entry$feature$featur;
      return (_entry$feature$featur = entry.feature[feature.id]) === null || _entry$feature$featur === void 0 ? void 0 : _entry$feature$featur.includes(privilege.id);
    }));
    const hasSupersededBasePrivileges = hasAssignedBasePrivileges && this.kibanaPrivileges.getBasePrivileges(entry).some(privilege => globalPrivileges.grantsPrivilege(privilege) && !formPrivileges.grantsPrivilege(privilege));
    const hasSupersededFeaturePrivileges = featuresWithDirectlyAssignedPrivileges.some(feature => feature.getAllPrivileges().some(fp => globalPrivileges.grantsPrivilege(fp) && !formPrivileges.grantsPrivilege(fp)));
    return hasSupersededBasePrivileges || hasSupersededFeaturePrivileges;
  }

  /**
   * Returns the *displayed* Primary Feature Privilege for the indicated feature and privilege index.
   * If the effective primary feature privilege is a "minimal" version, then this returns the corresponding non-minimal version.
   *
   * @example
   * The following kibana privilege entry will return `read`:
   * ```ts
   * const entry = {
   *    base: [],
   *    feature: {
   *       some_feature: ['minimal_read'],
   *    }
   * }
   * ```
   *
   * @param featureId the feature id to get the Primary Feature KibanaPrivilege for.
   * @param privilegeIndex the index of the kibana privileges role component
   * @param allSpacesSelected indicates if the privilege form is configured to grant access to all spaces.
   */
  getDisplayedPrimaryFeaturePrivilege(featureId, privilegeIndex, allSpacesSelected) {
    const feature = this.kibanaPrivileges.getSecuredFeature(featureId);
    const basePrivilege = this.getBasePrivilege(privilegeIndex);
    const selectedFeaturePrivileges = this.getSelectedFeaturePrivileges(featureId, privilegeIndex);
    const displayedPrivilege = feature.getPrimaryFeaturePrivileges().find(fp => {
      const correspondingMinimalPrivilegeId = fp.getMinimalPrivilegeId();
      const correspondingMinimalPrivilege = feature.getMinimalFeaturePrivileges().find(mp => mp.id === correspondingMinimalPrivilegeId);

      // There is only one case where the minimal privileges aren't available:
      // 1. Sub-feature privileges cannot be customized. When this is the case, the minimal privileges aren't registered with ES,
      // so they end up represented in the UI as an empty privilege. Empty privileges cannot be granted other privileges, so if we
      // encounter a minimal privilege that isn't granted by it's corresponding primary, then we know we've encountered this scenario.
      const hasMinimalPrivileges = fp.grantsPrivilege(correspondingMinimalPrivilege);
      return selectedFeaturePrivileges.includes(fp.id) || hasMinimalPrivileges && selectedFeaturePrivileges.includes(correspondingMinimalPrivilegeId) || (basePrivilege === null || basePrivilege === void 0 ? void 0 : basePrivilege.grantsPrivilege(fp));
    });
    const correctSpacesSelected = displayedPrivilege !== null && displayedPrivilege !== void 0 && displayedPrivilege.requireAllSpaces ? allSpacesSelected : true;
    const availablePrivileges = correctSpacesSelected && !(displayedPrivilege !== null && displayedPrivilege !== void 0 && displayedPrivilege.disabled);
    if (availablePrivileges) return displayedPrivilege;
  }
  getSelectedFeaturePrivileges(featureId, privilegeIndex) {
    var _this$role$kibana$pri;
    return (_this$role$kibana$pri = this.role.kibana[privilegeIndex].feature[featureId]) !== null && _this$role$kibana$pri !== void 0 ? _this$role$kibana$pri : [];
  }
  locateGlobalPrivilege(role) {
    return role.kibana.find(entry => (0, _privilege_utils.isGlobalPrivilegeDefinition)(entry));
  }
}
exports.PrivilegeFormCalculator = PrivilegeFormCalculator;