"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConversationalChain = ConversationalChain;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _prompts = require("@langchain/core/prompts");
var _runnables = require("@langchain/core/runnables");
var _output_parsers = require("@langchain/core/output_parsers");
var _ai = require("ai");
var _elasticsearch_retriever = require("./elasticsearch_retriever");
var _render_template = require("../utils/render_template");
var _get_citations = require("../utils/get_citations");
var _token_tracking = require("./token_tracking");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONDENSE_QUESTION_TEMPLATE = `Given the following conversation and a follow up question, rephrase the follow up question to be a standalone question, in its original language. Be verbose in your answer.

Chat History:
{chat_history}

Follow Up Input: {question}
Standalone question:`;
const condenseQuestionPrompt = _prompts.PromptTemplate.fromTemplate(CONDENSE_QUESTION_TEMPLATE);
const formatVercelMessages = chatHistory => {
  const formattedDialogueTurns = chatHistory.map(message => {
    if (message.role === 'user') {
      return `Human: ${message.content}`;
    } else if (message.role === 'assistant') {
      return `Assistant: ${message.content}`;
    } else {
      return `${message.role}: ${message.content}`;
    }
  });
  return formattedDialogueTurns.join('\n');
};
const buildContext = docs => {
  const serializedDocs = docs.map((doc, i) => (0, _render_template.renderTemplate)(`
position: ${i + 1}
{pageContent}`, {
    pageContent: doc.pageContent,
    ...doc.metadata
  }));
  return serializedDocs.join('\n');
};
class ConversationalChainFn {
  constructor(options) {
    (0, _defineProperty2.default)(this, "options", void 0);
    this.options = options;
  }
  async stream(client, msgs) {
    const data = new _ai.experimental_StreamData();
    const messages = msgs !== null && msgs !== void 0 ? msgs : [];
    const previousMessages = messages.slice(0, -1);
    const question = messages[messages.length - 1].content;
    const retrievedDocs = [];
    let retrievalChain = _runnables.RunnableLambda.from(() => '');
    if (this.options.rag) {
      var _this$options$rag$hit, _this$options$rag$siz;
      const retriever = new _elasticsearch_retriever.ElasticsearchRetriever({
        retriever: this.options.rag.retriever,
        index: this.options.rag.index,
        client: client.getClient(),
        content_field: this.options.rag.content_field,
        hit_doc_mapper: (_this$options$rag$hit = this.options.rag.hit_doc_mapper) !== null && _this$options$rag$hit !== void 0 ? _this$options$rag$hit : undefined,
        k: (_this$options$rag$siz = this.options.rag.size) !== null && _this$options$rag$siz !== void 0 ? _this$options$rag$siz : 3
      });
      retrievalChain = retriever.pipe(buildContext);
    }
    let standaloneQuestionChain = _runnables.RunnableLambda.from(input => input.question);
    if (previousMessages.length > 0) {
      standaloneQuestionChain = _runnables.RunnableSequence.from([condenseQuestionPrompt, this.options.model, new _output_parsers.StringOutputParser()]).withConfig({
        metadata: {
          type: 'standalone_question'
        }
      });
    }
    const prompt = _prompts.ChatPromptTemplate.fromTemplate(this.options.prompt);
    const answerChain = _runnables.RunnableSequence.from([{
      context: _runnables.RunnableSequence.from([input => input.question, retrievalChain]),
      chat_history: input => input.chat_history,
      question: input => input.question
    }, prompt, this.options.model.withConfig({
      metadata: {
        type: 'question_answer_qa'
      }
    })]);
    const conversationalRetrievalQAChain = _runnables.RunnableSequence.from([{
      question: standaloneQuestionChain,
      chat_history: input => input.chat_history
    }, answerChain, new _output_parsers.BytesOutputParser()]);
    const stream = await conversationalRetrievalQAChain.stream({
      question,
      chat_history: formatVercelMessages(previousMessages)
    }, {
      callbacks: [{
        handleChatModelStart(llm, msg, runId, parentRunId, extraParams, tags, metadata) {
          if ((metadata === null || metadata === void 0 ? void 0 : metadata.type) === 'question_answer_qa') {
            data.appendMessageAnnotation({
              type: 'prompt_token_count',
              count: (0, _token_tracking.getTokenEstimateFromMessages)(msg)
            });
          }
        },
        handleRetrieverEnd(documents) {
          retrievedDocs.push(...documents);
          data.appendMessageAnnotation({
            type: 'retrieved_docs',
            documents: documents
          });
          data.appendMessageAnnotation({
            type: 'context_token_count',
            count: (0, _token_tracking.getTokenEstimate)(buildContext(documents))
          });
        },
        handleChainEnd(outputs, runId, parentRunId) {
          var _outputs$constructor;
          if ((outputs === null || outputs === void 0 ? void 0 : (_outputs$constructor = outputs.constructor) === null || _outputs$constructor === void 0 ? void 0 : _outputs$constructor.name) === 'AIMessageChunk') {
            data.appendMessageAnnotation({
              type: 'citations',
              documents: (0, _get_citations.getCitations)(outputs.content, 'inline', retrievedDocs)
            });
          }

          // check that main chain (without parent) is finished:
          if (parentRunId == null) {
            data.close();
          }
        }
      }]
    });
    return stream.pipeThrough((0, _ai.createStreamDataTransformer)(true)).pipeThrough(data.stream);
  }
}
function ConversationalChain(options) {
  return new ConversationalChainFn(options);
}