"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseAgentSelection = exports.aggregateResults = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PER_PAGE = 9000;
const aggregateResults = async (generator, esClient, context) => {
  let results;
  const {
    results: initialResults,
    total
  } = await generator(1, PER_PAGE);
  const totalPages = Math.ceil(total / PER_PAGE);
  if (totalPages === 1) {
    // One page only, no need for PIT
    results = initialResults;
  } else {
    const {
      id: pitId
    } = await esClient.openPointInTime({
      index: _common.AGENTS_INDEX,
      keep_alive: '10m'
    });
    let currentSort;
    // Refetch first page with PIT
    const {
      results: pitInitialResults,
      searchAfter
    } = await generator(1, PER_PAGE, currentSort,
    // No searchAfter for first page, its built based on first page results
    pitId);
    results = pitInitialResults;
    currentSort = searchAfter;
    let currPage = 2;
    while (currPage <= totalPages) {
      const {
        results: additionalResults,
        searchAfter: additionalSearchAfter
      } = await generator(currPage++, PER_PAGE, currentSort, pitId);
      results.push(...additionalResults);
      currentSort = additionalSearchAfter;
    }
    try {
      await esClient.closePointInTime({
        id: pitId
      });
    } catch (error) {
      context.logFactory.get().warn(`Error closing point in time with id: ${pitId}. Error: ${error.message}`);
    }
  }
  return (0, _lodash.uniq)(results);
};
exports.aggregateResults = aggregateResults;
const parseAgentSelection = async (soClient, esClient, context, agentSelection) => {
  var _context$service$getA;
  const selectedAgents = new Set();
  const addAgent = selectedAgents.add.bind(selectedAgents);
  const {
    allAgentsSelected = false,
    platformsSelected = [],
    policiesSelected = [],
    agents = []
  } = agentSelection;
  const agentService = (_context$service$getA = context.service.getAgentService()) === null || _context$service$getA === void 0 ? void 0 : _context$service$getA.asInternalUser;
  const packagePolicyService = context.service.getPackagePolicyService();
  const kueryFragments = ['status:online'];
  if (agentService && packagePolicyService) {
    const osqueryPolicies = await aggregateResults(async (page, perPage) => {
      const {
        items,
        total
      } = await packagePolicyService.list(soClient, {
        kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${_common2.OSQUERY_INTEGRATION_NAME}`,
        perPage,
        page
      });
      return {
        results: items.map(it => it.policy_id),
        total
      };
    }, esClient, context);
    kueryFragments.push(`policy_id:(${(0, _lodash.uniq)(osqueryPolicies).join(' or ')})`);
    if (allAgentsSelected) {
      const kuery = kueryFragments.join(' and ');
      const fetchedAgents = await aggregateResults(async (page, perPage, searchAfter, pitId) => {
        const res = await agentService.listAgents({
          ...(searchAfter ? {
            searchAfter
          } : {}),
          ...(pitId ? {
            pitId
          } : {}),
          perPage,
          page,
          kuery,
          showInactive: false
        });
        return {
          results: res.agents.map(agent => agent.id),
          total: res.total,
          searchAfter: res.agents[res.agents.length - 1].sort
        };
      }, esClient, context);
      fetchedAgents.forEach(addAgent);
    } else {
      if (platformsSelected.length > 0 || policiesSelected.length > 0) {
        const groupFragments = [];
        if (platformsSelected.length) {
          groupFragments.push(`local_metadata.os.platform:(${platformsSelected.join(' or ')})`);
        }
        if (policiesSelected.length) {
          groupFragments.push(`policy_id:(${policiesSelected.join(' or ')})`);
        }
        kueryFragments.push(`(${groupFragments.join(' or ')})`);
        const kuery = kueryFragments.join(' and ');
        const fetchedAgents = await aggregateResults(async (page, perPage, searchAfter, pitId) => {
          const res = await agentService.listAgents({
            ...(searchAfter ? {
              searchAfter
            } : {}),
            ...(pitId ? {
              pitId
            } : {}),
            perPage,
            page,
            kuery,
            showInactive: false
          });
          return {
            results: res.agents.map(agent => agent.id),
            total: res.total,
            searchAfter: res.agents[res.agents.length - 1].sort
          };
        }, esClient, context);
        fetchedAgents.forEach(addAgent);
      }
    }
  }
  agents.forEach(addAgent);
  return Array.from(selectedAgents);
};
exports.parseAgentSelection = parseAgentSelection;