"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EQUATION_REGEX = void 0;
exports.validateCustomThreshold = validateCustomThreshold;
var _common = require("@kbn/data-plugin/common");
var _esQuery = require("@kbn/es-query");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _types = require("../../../../common/custom_threshold_rule/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EQUATION_REGEX = exports.EQUATION_REGEX = /[^A-Z|+|\-|\s|\d+|\.|\(|\)|\/|\*|>|<|=|\?|\:|&|\!|\|]+/g;
function validateCustomThreshold({
  criteria,
  searchConfiguration,
  uiSettings
}) {
  const validationResult = {
    errors: {}
  };
  const errors = {};
  validationResult.errors = errors;
  if (!searchConfiguration || !searchConfiguration.index) {
    errors.searchConfiguration = [_i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.invalidSearchConfiguration', {
      defaultMessage: 'Data view is required.'
    })];
  }
  if (searchConfiguration && searchConfiguration.query) {
    try {
      (0, _esQuery.buildEsQuery)(undefined, [{
        query: searchConfiguration.query.query,
        language: 'kuery'
      }], [], (0, _common.getEsQueryConfig)(uiSettings));
    } catch (e) {
      const errorReason = e.shortMessage || '';
      errors.filterQuery = [_i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.invalidFilterQuery', {
        values: {
          errorReason
        },
        defaultMessage: `Filter query is invalid. {errorReason}`
      })];
    }
  }
  if (!criteria || !criteria.length) {
    return validationResult;
  }
  criteria.forEach((c, idx) => {
    // Create an id for each criteria, so we can map errors to specific criteria.
    const id = idx.toString();
    errors[id] = errors[id] || {
      aggField: [],
      timeSizeUnit: [],
      timeWindowSize: [],
      critical: {
        threshold0: [],
        threshold1: []
      },
      metric: [],
      metrics: {}
    };
    if (!c.threshold || !c.threshold.length) {
      errors[id].critical.threshold0.push(_i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.thresholdRequired', {
        defaultMessage: 'Threshold is required.'
      }));
    }

    // The Threshold component returns an empty array with a length ([empty]) because it's using delete newThreshold[i].
    // We need to use [...c.threshold] to convert it to an array with an undefined value ([undefined]) so we can test each element.
    const {
      comparator,
      threshold
    } = {
      comparator: c.comparator,
      threshold: c.threshold
    };
    if (threshold && threshold.length && ![...threshold].every(isNumber)) {
      [...threshold].forEach((v, i) => {
        if (!isNumber(v)) {
          const key = i === 0 ? 'threshold0' : 'threshold1';
          errors[id].critical[key].push(_i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.thresholdTypeRequired', {
            defaultMessage: 'Thresholds must contain a valid number.'
          }));
        }
      });
    }
    if (comparator === _types.Comparator.BETWEEN && (!threshold || threshold.length < 2)) {
      errors[id].critical.threshold1.push(_i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.thresholdRequired', {
        defaultMessage: 'Threshold is required.'
      }));
    }
    if (!c.timeSize) {
      errors[id].timeWindowSize.push(_i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.timeRequred', {
        defaultMessage: 'Time size is Required.'
      }));
    }
    if (!c.metrics || c.metrics && c.metrics.length < 1) {
      errors[id].metricsError = _i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.metricsError', {
        defaultMessage: 'You must define at least 1 custom metric'
      });
    } else {
      c.metrics.forEach(metric => {
        const customMetricErrors = {};
        if (!metric.aggType) {
          customMetricErrors.aggType = _i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.metrics.aggTypeRequired', {
            defaultMessage: 'Aggregation is required'
          });
        }
        if (metric.aggType !== 'count' && !metric.field) {
          customMetricErrors.field = _i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.metrics.fieldRequired', {
            defaultMessage: 'Field is required'
          });
        }
        if (metric.aggType === 'count' && metric.filter) {
          try {
            (0, _esQuery.fromKueryExpression)(metric.filter);
          } catch (e) {
            customMetricErrors.filter = e.message;
          }
        }
        if (!(0, _lodash.isEmpty)(customMetricErrors)) {
          errors[id].metrics[metric.name] = customMetricErrors;
        }
      });
    }
    if (c.equation && c.equation.match(EQUATION_REGEX)) {
      errors[id].equation = _i18n.i18n.translate('xpack.observability.customThreshold.rule.alertFlyout.error.equation.invalidCharacters', {
        defaultMessage: 'The equation field only supports the following characters: A-Z, +, -, /, *, (, ), ?, !, &, :, |, >, <, ='
      });
    }
  });
  return validationResult;
}
const isNumber = value => typeof value === 'number';