"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityAIAssistantClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = require("@hapi/boom");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _connectors = require("../../../common/connectors");
var _conversation_complete = require("../../../common/conversation_complete");
var _without_token_count_events = require("../../../common/utils/without_token_count_events");
var _knowledge_base_service = require("../knowledge_base_service");
var _get_access_query = require("../util/get_access_query");
var _get_system_message_from_instructions = require("../util/get_system_message_from_instructions");
var _replace_system_message = require("../util/replace_system_message");
var _with_assistant_span = require("../util/with_assistant_span");
var _bedrock_claude_adapter = require("./adapters/bedrock/bedrock_claude_adapter");
var _fail_on_non_existing_function_call = require("./adapters/fail_on_non_existing_function_call");
var _openai_adapter = require("./adapters/openai_adapter");
var _get_context_function_request_if_needed = require("./get_context_function_request_if_needed");
var _extract_messages = require("./operators/extract_messages");
var _extract_token_count = require("./operators/extract_token_count");
var _instrument_and_count_tokens = require("./operators/instrument_and_count_tokens");
var _continue_conversation = require("./operators/continue_conversation");
var _get_generated_title = require("./operators/get_generated_title");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_FUNCTION_CALLS = 8;
class ObservabilityAIAssistantClient {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "getConversationWithMetaFields", async conversationId => {
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: this.dependencies.resources.aliases.conversations,
        query: {
          bool: {
            filter: [...(0, _get_access_query.getAccessQuery)({
              user: this.dependencies.user,
              namespace: this.dependencies.namespace
            }), {
              term: {
                'conversation.id': conversationId
              }
            }]
          }
        },
        size: 1,
        terminate_after: 1
      });
      return response.hits.hits[0];
    });
    (0, _defineProperty2.default)(this, "getConversationUpdateValues", lastUpdated => {
      return {
        conversation: {
          last_updated: lastUpdated
        },
        user: this.dependencies.user,
        namespace: this.dependencies.namespace
      };
    });
    (0, _defineProperty2.default)(this, "get", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      return conversation._source;
    });
    (0, _defineProperty2.default)(this, "delete", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      await this.dependencies.esClient.asInternalUser.delete({
        id: conversation._id,
        index: conversation._index,
        refresh: true
      });
    });
    (0, _defineProperty2.default)(this, "complete", params => {
      const {
        functionClient,
        connectorId,
        simulateFunctionCalling,
        instructions: requestInstructions = [],
        messages: initialMessages,
        signal,
        responseLanguage = 'English',
        persist,
        kibanaPublicUrl,
        isPublic,
        title: predefinedTitle,
        conversationId: predefinedConversationId,
        disableFunctions = false
      } = params;
      if (responseLanguage) {
        requestInstructions.push(`You MUST respond in the users preferred language which is: ${responseLanguage}.`);
      }
      const isConversationUpdate = persist && !!predefinedConversationId;
      const conversationId = persist ? predefinedConversationId || (0, _uuid.v4)() : '';
      if (persist && !isConversationUpdate && kibanaPublicUrl) {
        requestInstructions.push(`This conversation will be persisted in Kibana and available at this url: ${kibanaPublicUrl + `/app/observabilityAIAssistant/conversations/${conversationId}`}.`);
      }
      const kbInstructions$ = (0, _rxjs.from)(this.fetchKnowledgeBaseInstructions()).pipe((0, _rxjs.shareReplay)());

      // from the initial messages, override any system message with
      // the one that is based on the instructions (registered, request, kb)
      const messagesWithUpdatedSystemMessage$ = kbInstructions$.pipe((0, _rxjs.map)(knowledgeBaseInstructions => {
        // this is what we eventually store in the conversation
        const messagesWithUpdatedSystemMessage = (0, _replace_system_message.replaceSystemMessage)((0, _get_system_message_from_instructions.getSystemMessageFromInstructions)({
          registeredInstructions: functionClient.getInstructions(),
          knowledgeBaseInstructions,
          requestInstructions,
          availableFunctionNames: functionClient.getFunctions().map(fn => fn.definition.name)
        }), initialMessages);
        return messagesWithUpdatedSystemMessage;
      }), (0, _rxjs.shareReplay)());

      // if it is:
      // - a new conversation
      // - no predefined title is given
      // - we need to store the conversation
      // we generate a title
      // if not, we complete with an empty string
      const title$ = predefinedTitle || isConversationUpdate || !persist ? (0, _rxjs.of)(predefinedTitle || '').pipe((0, _rxjs.shareReplay)()) : messagesWithUpdatedSystemMessage$.pipe((0, _rxjs.switchMap)(messages => (0, _get_generated_title.getGeneratedTitle)({
        messages,
        responseLanguage,
        logger: this.dependencies.logger,
        chat: (name, chatParams) => {
          return this.chat(name, {
            ...chatParams,
            simulateFunctionCalling,
            connectorId,
            signal
          });
        }
      })), (0, _rxjs.shareReplay)());

      // we continue the conversation here, after resolving both the materialized
      // messages and the knowledge base instructions
      const nextEvents$ = (0, _rxjs.combineLatest)([messagesWithUpdatedSystemMessage$, kbInstructions$]).pipe((0, _rxjs.switchMap)(([messagesWithUpdatedSystemMessage, knowledgeBaseInstructions]) => {
        // if needed, inject a context function request here
        const contextRequest = functionClient.hasFunction('context') ? (0, _get_context_function_request_if_needed.getContextFunctionRequestIfNeeded)(messagesWithUpdatedSystemMessage) : undefined;
        return (0, _rxjs.merge)(
        // if we have added a context function request, also emit
        // the messageAdd event for it, so we can notify the consumer
        // and add it to the conversation
        ...(contextRequest ? [(0, _rxjs.of)(contextRequest)] : []), (0, _continue_conversation.continueConversation)({
          messages: [...messagesWithUpdatedSystemMessage, ...(contextRequest ? [contextRequest.message] : [])],
          chat: (name, chatParams) => {
            // inject a chat function with predefined parameters
            return this.chat(name, {
              ...chatParams,
              signal,
              simulateFunctionCalling,
              connectorId
            });
          },
          // start out with the max number of function calls
          functionCallsLeft: MAX_FUNCTION_CALLS,
          functionClient,
          knowledgeBaseInstructions,
          requestInstructions,
          signal,
          logger: this.dependencies.logger,
          disableFunctions
        }));
      }), (0, _rxjs.shareReplay)());
      const output$ = (0, _rxjs.merge)(
      // get all the events from continuing the conversation
      nextEvents$,
      // wait until all dependencies have completed
      (0, _rxjs.forkJoin)([messagesWithUpdatedSystemMessage$,
      // get just the new messages
      nextEvents$.pipe((0, _without_token_count_events.withoutTokenCountEvents)(), (0, _extract_messages.extractMessages)()),
      // count all the token count events emitted during completion
      (0, _rxjs.merge)(nextEvents$, title$.pipe((0, _rxjs.filter)(value => typeof value !== 'string'))).pipe((0, _extract_token_count.extractTokenCount)()),
      // get just the title, and drop the token count events
      title$.pipe((0, _rxjs.filter)(value => typeof value === 'string'))]).pipe((0, _rxjs.switchMap)(([messagesWithUpdatedSystemMessage, addedMessages, tokenCountResult, title]) => {
        var _lastMessage$message$;
        const initialMessagesWithAddedMessages = messagesWithUpdatedSystemMessage.concat(addedMessages);
        const lastMessage = initialMessagesWithAddedMessages[initialMessagesWithAddedMessages.length - 1];

        // if a function request is at the very end, close the stream to consumer
        // without persisting or updating the conversation. we need to wait
        // on the function response to have a valid conversation
        const isFunctionRequest = (_lastMessage$message$ = lastMessage.message.function_call) === null || _lastMessage$message$ === void 0 ? void 0 : _lastMessage$message$.name;
        if (!persist || isFunctionRequest) {
          return (0, _rxjs.of)();
        }
        if (isConversationUpdate) {
          return (0, _rxjs.from)(this.getConversationWithMetaFields(conversationId)).pipe((0, _rxjs.switchMap)(conversation => {
            var _conversation$_source, _conversation$_source2, _conversation$_source3;
            if (!conversation) {
              return (0, _rxjs.throwError)(() => (0, _conversation_complete.createConversationNotFoundError)());
            }
            const persistedTokenCount = (_conversation$_source = (_conversation$_source2 = conversation._source) === null || _conversation$_source2 === void 0 ? void 0 : _conversation$_source2.conversation.token_count) !== null && _conversation$_source !== void 0 ? _conversation$_source : {
              prompt: 0,
              completion: 0,
              total: 0
            };
            return (0, _rxjs.from)(this.update(conversationId, (0, _lodash.merge)({},
            // base conversation without messages
            (0, _lodash.omit)(conversation._source, 'messages'),
            // update messages
            {
              messages: initialMessagesWithAddedMessages
            },
            // update token count
            {
              conversation: {
                title: title || ((_conversation$_source3 = conversation._source) === null || _conversation$_source3 === void 0 ? void 0 : _conversation$_source3.conversation.title),
                token_count: {
                  prompt: persistedTokenCount.prompt + tokenCountResult.prompt,
                  completion: persistedTokenCount.completion + tokenCountResult.completion,
                  total: persistedTokenCount.total + tokenCountResult.total
                }
              }
            })));
          })).pipe((0, _rxjs.map)(conversation => {
            return {
              conversation: conversation.conversation,
              type: _conversation_complete.StreamingChatResponseEventType.ConversationUpdate
            };
          }));
        }
        return (0, _rxjs.from)(this.create({
          '@timestamp': new Date().toISOString(),
          conversation: {
            title,
            id: conversationId,
            token_count: tokenCountResult
          },
          public: !!isPublic,
          labels: {},
          numeric_labels: {},
          messages: initialMessagesWithAddedMessages
        })).pipe((0, _rxjs.map)(conversation => {
          return {
            conversation: conversation.conversation,
            type: _conversation_complete.StreamingChatResponseEventType.ConversationCreate
          };
        }));
      })));
      return output$.pipe((0, _instrument_and_count_tokens.instrumentAndCountTokens)('complete'), (0, _without_token_count_events.withoutTokenCountEvents)(), (0, _rxjs.catchError)(error => {
        this.dependencies.logger.error(error);
        return (0, _rxjs.throwError)(() => error);
      }), (0, _rxjs.tap)(event => {
        if (this.dependencies.logger.isLevelEnabled('debug')) {
          switch (event.type) {
            case _conversation_complete.StreamingChatResponseEventType.MessageAdd:
              this.dependencies.logger.debug(`Added message: ${JSON.stringify(event.message)}`);
              break;
            case _conversation_complete.StreamingChatResponseEventType.ConversationCreate:
              this.dependencies.logger.debug(`Created conversation: ${JSON.stringify(event.conversation)}`);
              break;
            case _conversation_complete.StreamingChatResponseEventType.ConversationUpdate:
              this.dependencies.logger.debug(`Updated conversation: ${JSON.stringify(event.conversation)}`);
              break;
          }
        }
      }), (0, _rxjs.shareReplay)());
    });
    (0, _defineProperty2.default)(this, "chat", (name, {
      messages,
      connectorId,
      functions,
      functionCall,
      signal,
      simulateFunctionCalling
    }) => {
      return (0, _rxjs.defer)(() => (0, _rxjs.from)((0, _with_assistant_span.withAssistantSpan)('get_connector', () => this.dependencies.actionsClient.get({
        id: connectorId,
        throwIfSystemAction: true
      })))).pipe((0, _rxjs.switchMap)(connector => {
        this.dependencies.logger.debug(`Creating "${connector.actionTypeId}" adapter`);
        let adapter;
        switch (connector.actionTypeId) {
          case _connectors.ObservabilityAIAssistantConnectorType.OpenAI:
            adapter = (0, _openai_adapter.createOpenAiAdapter)({
              messages,
              functions,
              functionCall,
              logger: this.dependencies.logger,
              simulateFunctionCalling
            });
            break;
          case _connectors.ObservabilityAIAssistantConnectorType.Bedrock:
            adapter = (0, _bedrock_claude_adapter.createBedrockClaudeAdapter)({
              messages,
              functions,
              functionCall,
              logger: this.dependencies.logger
            });
            break;
          default:
            throw new Error(`Connector type is not supported: ${connector.actionTypeId}`);
        }
        const subAction = adapter.getSubAction();
        this.dependencies.logger.trace(JSON.stringify(subAction.subActionParams, null, 2));
        return (0, _rxjs.from)((0, _with_assistant_span.withAssistantSpan)('get_execute_result', () => this.dependencies.actionsClient.execute({
          actionId: connectorId,
          params: subAction
        }))).pipe((0, _rxjs.switchMap)(executeResult => {
          if (executeResult.status === 'error' && executeResult !== null && executeResult !== void 0 && executeResult.serviceMessage) {
            const tokenLimitRegex = /This model's maximum context length is (\d+) tokens\. However, your messages resulted in (\d+) tokens/g;
            const tokenLimitRegexResult = tokenLimitRegex.exec(executeResult.serviceMessage);
            if (tokenLimitRegexResult) {
              const [, tokenLimit, tokenCount] = tokenLimitRegexResult;
              throw (0, _conversation_complete.createTokenLimitReachedError)(parseInt(tokenLimit, 10), parseInt(tokenCount, 10));
            }
          }
          if (executeResult.status === 'error') {
            throw (0, _conversation_complete.createInternalServerError)(`${executeResult === null || executeResult === void 0 ? void 0 : executeResult.message} - ${executeResult === null || executeResult === void 0 ? void 0 : executeResult.serviceMessage}`);
          }
          const response = executeResult.data;
          signal.addEventListener('abort', () => response.destroy());
          return adapter.streamIntoObservable(response);
        }));
      }), (0, _instrument_and_count_tokens.instrumentAndCountTokens)(name), (0, _fail_on_non_existing_function_call.failOnNonExistingFunctionCall)({
        functions
      }), (0, _rxjs.tap)(event => {
        if (event.type === _conversation_complete.StreamingChatResponseEventType.ChatCompletionChunk && this.dependencies.logger.isLevelEnabled('trace')) {
          this.dependencies.logger.trace(`Received chunk: ${JSON.stringify(event.message)}`);
        }
      }), (0, _rxjs.shareReplay)());
    });
    (0, _defineProperty2.default)(this, "find", async options => {
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: this.dependencies.resources.aliases.conversations,
        allow_no_indices: true,
        query: {
          bool: {
            filter: [...(0, _get_access_query.getAccessQuery)({
              user: this.dependencies.user,
              namespace: this.dependencies.namespace
            })]
          }
        },
        sort: {
          '@timestamp': 'desc'
        },
        size: 100
      });
      return {
        conversations: response.hits.hits.map(hit => hit._source)
      };
    });
    (0, _defineProperty2.default)(this, "update", async (conversationId, conversation) => {
      const persistedConversation = await this.getConversationWithMetaFields(conversationId);
      if (!persistedConversation) {
        throw (0, _boom.notFound)();
      }
      const updatedConversation = (0, _lodash.merge)({}, conversation, this.getConversationUpdateValues(new Date().toISOString()));
      await this.dependencies.esClient.asInternalUser.update({
        id: persistedConversation._id,
        index: persistedConversation._index,
        doc: updatedConversation,
        refresh: true
      });
      return updatedConversation;
    });
    (0, _defineProperty2.default)(this, "setTitle", async ({
      conversationId,
      title
    }) => {
      const document = await this.getConversationWithMetaFields(conversationId);
      if (!document) {
        throw (0, _boom.notFound)();
      }
      const conversation = await this.get(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      const updatedConversation = (0, _lodash.merge)({}, conversation, {
        conversation: {
          title
        }
      }, this.getConversationUpdateValues(new Date().toISOString()));
      await this.dependencies.esClient.asInternalUser.update({
        id: document._id,
        index: document._index,
        doc: {
          conversation: {
            title
          }
        },
        refresh: true
      });
      return updatedConversation;
    });
    (0, _defineProperty2.default)(this, "create", async conversation => {
      const now = new Date().toISOString();
      const createdConversation = (0, _lodash.merge)({}, conversation, {
        '@timestamp': now,
        conversation: {
          id: conversation.conversation.id || (0, _uuid.v4)()
        }
      }, this.getConversationUpdateValues(now));
      await this.dependencies.esClient.asInternalUser.index({
        index: this.dependencies.resources.aliases.conversations,
        document: createdConversation,
        refresh: true
      });
      return createdConversation;
    });
    (0, _defineProperty2.default)(this, "recall", async ({
      queries,
      categories
    }) => {
      return this.dependencies.knowledgeBaseService.recall({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        queries,
        categories,
        asCurrentUser: this.dependencies.esClient.asCurrentUser
      });
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseStatus", () => {
      return this.dependencies.knowledgeBaseService.status();
    });
    (0, _defineProperty2.default)(this, "setupKnowledgeBase", () => {
      return this.dependencies.knowledgeBaseService.setup();
    });
    (0, _defineProperty2.default)(this, "createKnowledgeBaseEntry", async ({
      entry
    }) => {
      return this.dependencies.knowledgeBaseService.addEntry({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        entry
      });
    });
    (0, _defineProperty2.default)(this, "importKnowledgeBaseEntries", async ({
      entries
    }) => {
      const operations = entries.map(entry => ({
        type: _knowledge_base_service.KnowledgeBaseEntryOperationType.Index,
        document: {
          ...entry,
          '@timestamp': new Date().toISOString()
        }
      }));
      await this.dependencies.knowledgeBaseService.addEntries({
        operations
      });
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseEntries", async ({
      query,
      sortBy,
      sortDirection
    }) => {
      return this.dependencies.knowledgeBaseService.getEntries({
        query,
        sortBy,
        sortDirection
      });
    });
    (0, _defineProperty2.default)(this, "deleteKnowledgeBaseEntry", async id => {
      return this.dependencies.knowledgeBaseService.deleteEntry({
        id
      });
    });
    (0, _defineProperty2.default)(this, "fetchKnowledgeBaseInstructions", async () => {
      const knowledgeBaseInstructions = await this.dependencies.knowledgeBaseService.getInstructions(this.dependencies.namespace, this.dependencies.user);
      return knowledgeBaseInstructions;
    });
    this.dependencies = dependencies;
  }
}
exports.ObservabilityAIAssistantClient = ObservabilityAIAssistantClient;