"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOpenAiAdapter = void 0;
var _gptTokenizer = require("gpt-tokenizer");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _common = require("../../../../common");
var _process_openai_stream = require("./process_openai_stream");
var _eventsource_stream_into_observable = require("../../util/eventsource_stream_into_observable");
var _parse_inline_function_calls = require("./simulate_function_calling/parse_inline_function_calls");
var _get_messages_with_simulated_function_calling = require("./simulate_function_calling/get_messages_with_simulated_function_calling");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getOpenAIPromptTokenCount({
  messages,
  functions
}) {
  // per https://github.com/openai/openai-cookbook/blob/main/examples/How_to_count_tokens_with_tiktoken.ipynb
  const tokensFromMessages = (0, _gptTokenizer.encode)(messages.map(({
    message
  }) => `<|start|>${message.role}\n${message.content}\n${'name' in message ? message.name : 'function_call' in message && message.function_call ? message.function_call.name + '\n' + message.function_call.arguments : ''}<|end|>`).join('\n')).length;

  // this is an approximation. OpenAI cuts off a function schema
  // at a certain level of nesting, so their token count might
  // be lower than what we are calculating here.
  const tokensFromFunctions = functions ? (0, _gptTokenizer.encode)(functions === null || functions === void 0 ? void 0 : functions.map(fn => `<|start|>${fn.name}\n${fn.description}\n${JSON.stringify(fn.parameters)}<|end|>`).join('\n')).length : 0;
  return tokensFromMessages + tokensFromFunctions;
}
function messagesToOpenAI(messages) {
  return (0, _lodash.compact)(messages.filter(message => {
    var _message$message$func;
    return message.message.content || ((_message$message$func = message.message.function_call) === null || _message$message$func === void 0 ? void 0 : _message$message$func.name);
  }).map(message => {
    var _message$message$func2;
    const role = message.message.role === _common.MessageRole.Elastic ? _common.MessageRole.User : message.message.role;
    return {
      role,
      content: message.message.content,
      function_call: (0, _lodash.isEmpty)((_message$message$func2 = message.message.function_call) === null || _message$message$func2 === void 0 ? void 0 : _message$message$func2.name) ? undefined : (0, _lodash.omit)(message.message.function_call, 'trigger'),
      name: message.message.name
    };
  }));
}
const createOpenAiAdapter = ({
  messages,
  functions,
  functionCall,
  logger,
  simulateFunctionCalling
}) => {
  const promptTokens = getOpenAIPromptTokenCount({
    messages,
    functions
  });
  return {
    getSubAction: () => {
      const functionsForOpenAI = functions === null || functions === void 0 ? void 0 : functions.map(fn => ({
        ...fn,
        parameters: (0, _lodash.merge)({
          type: 'object',
          properties: {}
        }, fn.parameters)
      }));
      let request;
      if (simulateFunctionCalling) {
        request = {
          messages: messagesToOpenAI((0, _get_messages_with_simulated_function_calling.getMessagesWithSimulatedFunctionCalling)({
            messages,
            functions: functionsForOpenAI,
            functionCall
          })),
          stream: true,
          temperature: 0
        };
      } else {
        request = {
          messages: messagesToOpenAI(messages),
          stream: true,
          ...(!!(functionsForOpenAI !== null && functionsForOpenAI !== void 0 && functionsForOpenAI.length) ? {
            tools: functionsForOpenAI.map(fn => ({
              function: (0, _lodash.pick)(fn, 'name', 'description', 'parameters'),
              type: 'function'
            }))
          } : {}),
          temperature: 0,
          tool_choice: functionCall ? {
            function: {
              name: functionCall
            },
            type: 'function'
          } : undefined
        };
      }
      return {
        subAction: 'stream',
        subActionParams: {
          body: JSON.stringify(request),
          stream: true
        }
      };
    },
    streamIntoObservable: readable => {
      return (0, _eventsource_stream_into_observable.eventsourceStreamIntoObservable)(readable).pipe((0, _process_openai_stream.processOpenAiStream)({
        promptTokenCount: promptTokens,
        logger
      }), simulateFunctionCalling ? (0, _parse_inline_function_calls.parseInlineFunctionCalls)({
        logger
      }) : _rxjs.identity);
    }
  };
};
exports.createOpenAiAdapter = createOpenAiAdapter;