"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processBedrockStream = processBedrockStream;
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _conversation_complete = require("../../../../../common/conversation_complete");
var _parse_serde_chunk_body = require("./parse_serde_chunk_body");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function processBedrockStream() {
  return source => new _rxjs.Observable(subscriber => {
    const id = (0, _uuid.v4)();

    // We use this to make sure we don't complete the Observable
    // before all operations have completed.
    let nextPromise = Promise.resolve();

    // As soon as we see a `<function` token, we write all chunks
    // to a buffer, that we flush as a function request if we
    // spot the stop sequence.

    async function handleNext(value) {
      const chunkBody = (0, _parse_serde_chunk_body.parseSerdeChunkBody)(value.chunk);
      if (isTokenCountCompletionChunk(chunkBody)) {
        return emitTokenCountEvent(subscriber, chunkBody);
      }
      if (chunkBody.type !== 'content_block_start' && chunkBody.type !== 'content_block_delta' && chunkBody.type !== 'message_delta') {
        return;
      }

      // completion: what we eventually want to emit
      const completion = chunkBody.type !== 'message_delta' ? getCompletion(chunkBody) : chunkBody.delta.stop_sequence || '';
      subscriber.next({
        id,
        type: _conversation_complete.StreamingChatResponseEventType.ChatCompletionChunk,
        message: {
          content: completion
        }
      });
    }
    source.subscribe({
      next: value => {
        nextPromise = nextPromise.then(() => handleNext(value).catch(error => subscriber.error(error)));
      },
      error: err => {
        subscriber.error(err);
      },
      complete: () => {
        nextPromise.then(() => subscriber.complete());
      }
    });
  });
}
function isTokenCountCompletionChunk(value) {
  return value.type === 'message_stop' && 'amazon-bedrock-invocationMetrics' in value;
}
function emitTokenCountEvent(subscriber, chunk) {
  const {
    inputTokenCount,
    outputTokenCount
  } = chunk['amazon-bedrock-invocationMetrics'];
  subscriber.next({
    type: _conversation_complete.StreamingChatResponseEventType.TokenCount,
    tokens: {
      completion: outputTokenCount,
      prompt: inputTokenCount,
      total: inputTokenCount + outputTokenCount
    }
  });
}
function getCompletion(chunk) {
  return chunk.type === 'content_block_start' ? chunk.content_block.text : chunk.delta.text;
}