"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FunctionArgsValidationError = exports.ChatFunctionClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ajv = _interopRequireDefault(require("ajv"));
var _dedent = _interopRequireDefault(require("dedent"));
var _lodash = require("lodash");
var _types = require("../../../common/functions/types");
var _filter_function_definitions = require("../../../common/utils/filter_function_definitions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/* eslint-disable max-classes-per-file*/

class FunctionArgsValidationError extends Error {
  constructor(errors) {
    super('Function arguments are invalid');
    this.errors = errors;
  }
}
exports.FunctionArgsValidationError = FunctionArgsValidationError;
const ajv = new _ajv.default({
  strict: false
});
class ChatFunctionClient {
  constructor(screenContexts) {
    (0, _defineProperty2.default)(this, "instructions", []);
    (0, _defineProperty2.default)(this, "functionRegistry", new Map());
    (0, _defineProperty2.default)(this, "validators", new Map());
    (0, _defineProperty2.default)(this, "actions", void 0);
    (0, _defineProperty2.default)(this, "registerFunction", (definition, respond) => {
      if (definition.parameters) {
        this.validators.set(definition.name, ajv.compile(definition.parameters));
      }
      this.functionRegistry.set(definition.name, {
        definition,
        respond
      });
    });
    (0, _defineProperty2.default)(this, "registerInstruction", instruction => {
      this.instructions.push(instruction);
    });
    this.screenContexts = screenContexts;
    const allData = (0, _lodash.compact)(screenContexts.flatMap(context => context.data));
    this.actions = (0, _lodash.compact)(screenContexts.flatMap(context => context.actions));
    if (allData.length) {
      this.registerFunction({
        name: 'get_data_on_screen',
        description: (0, _dedent.default)(`Get data that is on the screen:
            ${allData.map(data => `${data.name}: ${data.description}`).join('\n')}
          `),
        visibility: _types.FunctionVisibility.AssistantOnly,
        parameters: {
          type: 'object',
          properties: {
            data: {
              type: 'array',
              description: 'The pieces of data you want to look at it. You can request one, or multiple',
              items: {
                type: 'string',
                enum: allData.map(data => data.name)
              }
            }
          },
          required: ['data']
        }
      }, async ({
        arguments: {
          data: dataNames
        }
      }) => {
        return {
          content: allData.filter(data => dataNames.includes(data.name))
        };
      });
    }
    this.actions.forEach(action => {
      if (action.parameters) {
        this.validators.set(action.name, ajv.compile(action.parameters));
      }
    });
  }
  validate(name, parameters) {
    const validator = this.validators.get(name);
    if (!validator) {
      return;
    }
    const result = validator(parameters);
    if (!result) {
      throw new FunctionArgsValidationError(validator.errors);
    }
  }
  getInstructions() {
    return this.instructions;
  }
  hasAction(name) {
    return !!this.actions.find(action => action.name === name);
  }
  getFunctions({
    filter
  } = {}) {
    const allFunctions = Array.from(this.functionRegistry.values());
    const functionsByName = (0, _lodash.keyBy)(allFunctions, definition => definition.definition.name);
    const matchingDefinitions = (0, _filter_function_definitions.filterFunctionDefinitions)({
      filter,
      definitions: allFunctions.map(fn => fn.definition)
    });
    return matchingDefinitions.map(definition => functionsByName[definition.name]);
  }
  getActions() {
    return this.actions;
  }
  hasFunction(name) {
    return this.functionRegistry.has(name);
  }
  async executeFunction({
    chat,
    name,
    args,
    messages,
    signal
  }) {
    const fn = this.functionRegistry.get(name);
    if (!fn) {
      throw new Error(`Function ${name} not found`);
    }
    const parsedArguments = args ? JSON.parse(args) : {};
    this.validate(name, parsedArguments);
    return await fn.respond({
      arguments: parsedArguments,
      messages,
      screenContexts: this.screenContexts,
      chat
    }, signal);
  }
}
exports.ChatFunctionClient = ChatFunctionClient;