"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createChatService = createChatService;
var _common = require("@kbn/kibana-utils-plugin/common");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _common2 = require("../../common");
var _conversation_complete = require("../../common/conversation_complete");
var _types = require("../../common/functions/types");
var _filter_function_definitions = require("../../common/utils/filter_function_definitions");
var _throw_serialized_chat_completion_errors = require("../../common/utils/throw_serialized_chat_completion_errors");
var _analytics = require("../analytics");
var _readable_stream_reader_into_observable = require("../utils/readable_stream_reader_into_observable");
var _complete = require("./complete");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MIN_DELAY = 10;
function toObservable(response) {
  var _response$response, _response$response$bo;
  const status = (_response$response = response.response) === null || _response$response === void 0 ? void 0 : _response$response.status;
  if (!status || status >= 400) {
    var _response$response2;
    throw new Error(((_response$response2 = response.response) === null || _response$response2 === void 0 ? void 0 : _response$response2.statusText) || 'Unexpected error');
  }
  const reader = (_response$response$bo = response.response.body) === null || _response$response$bo === void 0 ? void 0 : _response$response$bo.getReader();
  if (!reader) {
    throw new Error('Could not get reader from response');
  }
  return (0, _readable_stream_reader_into_observable.readableStreamReaderIntoObservable)(reader).pipe(
  // append a timestamp of when each value was emitted
  (0, _rxjs.timestamp)(),
  // use the previous timestamp to calculate a target
  // timestamp for emitting the next value
  (0, _rxjs.scan)((acc, value) => {
    const lastTimestamp = acc.timestamp || 0;
    const emitAt = Math.max(lastTimestamp + MIN_DELAY, value.timestamp);
    return {
      timestamp: emitAt,
      value: value.value
    };
  }),
  // add the delay based on the elapsed time
  // using concatMap(of(value).pipe(delay(50))
  // leads to browser issues because timers
  // are throttled when the tab is not active
  (0, _rxjs.concatMap)(value => {
    const now = Date.now();
    const delayFor = value.timestamp - now;
    if (delayFor <= 0) {
      return (0, _rxjs.of)(value.value);
    }
    return (0, _rxjs.of)(value.value).pipe((0, _rxjs.delay)(delayFor));
  }));
}
async function createChatService({
  analytics,
  signal: setupAbortSignal,
  registrations,
  apiClient
}) {
  const functionRegistry = new Map();
  const renderFunctionRegistry = new Map();
  const [{
    functionDefinitions,
    systemMessage
  }] = await Promise.all([apiClient('GET /internal/observability_ai_assistant/functions', {
    signal: setupAbortSignal
  }), ...registrations.map(registration => {
    return registration({
      registerRenderFunction: (name, renderFn) => {
        renderFunctionRegistry.set(name, renderFn);
      }
    });
  })]);
  functionDefinitions.forEach(fn => {
    functionRegistry.set(fn.name, fn);
  });
  const getFunctions = options => {
    return (0, _filter_function_definitions.filterFunctionDefinitions)({
      ...options,
      definitions: functionDefinitions
    });
  };
  const client = {
    chat(name, {
      connectorId,
      messages,
      function: callFunctions = 'auto',
      signal
    }) {
      return new _rxjs.Observable(subscriber => {
        const functions = getFunctions().filter(fn => {
          var _fn$visibility;
          const visibility = (_fn$visibility = fn.visibility) !== null && _fn$visibility !== void 0 ? _fn$visibility : _types.FunctionVisibility.All;
          return visibility === _types.FunctionVisibility.All || visibility === _types.FunctionVisibility.AssistantOnly;
        });
        apiClient('POST /internal/observability_ai_assistant/chat', {
          params: {
            body: {
              name,
              messages,
              connectorId,
              functions: callFunctions === 'none' ? [] : functions.map(fn => (0, _lodash.pick)(fn, 'name', 'description', 'parameters'))
            }
          },
          signal,
          asResponse: true,
          rawResponse: true
        }).then(_response => {
          const response = _response;
          const subscription = toObservable(response).pipe((0, _rxjs.map)(line => JSON.parse(line)), (0, _rxjs.filter)(line => line.type !== _conversation_complete.StreamingChatResponseEventType.BufferFlush && line.type !== _conversation_complete.StreamingChatResponseEventType.TokenCount), (0, _throw_serialized_chat_completion_errors.throwSerializedChatCompletionErrors)()).subscribe(subscriber);

          // if the request is aborted, convert that into state as well
          signal.addEventListener('abort', () => {
            subscriber.error(new _common.AbortError());
            subscription.unsubscribe();
          });
        }).catch(async err => {
          if ('response' in err) {
            var _err$response;
            const body = await ((_err$response = err.response) === null || _err$response === void 0 ? void 0 : _err$response.json());
            err.body = body;
            if (body.message) {
              err.message = body.message;
            }
          }
          throw err;
        }).catch(err => {
          subscriber.error(err);
        });
        return subscriber;
      }).pipe(
      // make sure the request is only triggered once,
      // even with multiple subscribers
      (0, _rxjs.shareReplay)());
    },
    complete({
      getScreenContexts,
      connectorId,
      conversationId,
      messages,
      persist,
      disableFunctions,
      signal,
      responseLanguage
    }) {
      return (0, _complete.complete)({
        getScreenContexts,
        connectorId,
        conversationId,
        messages,
        persist,
        disableFunctions,
        signal,
        client,
        responseLanguage
      }, ({
        params
      }) => {
        return (0, _rxjs.from)(apiClient('POST /internal/observability_ai_assistant/chat/complete', {
          params,
          signal,
          asResponse: true,
          rawResponse: true
        })).pipe((0, _rxjs.map)(_response => toObservable(_response)), (0, _rxjs.switchMap)(response$ => response$), (0, _rxjs.map)(line => JSON.parse(line)), (0, _rxjs.shareReplay)());
      });
    }
  };
  return {
    sendAnalyticsEvent: event => {
      (0, _analytics.sendEvent)(analytics, event);
    },
    renderFunction: (name, args, response, onActionClick) => {
      var _response$content, _response$data;
      const fn = renderFunctionRegistry.get(name);
      if (!fn) {
        throw new Error(`Function ${name} not found`);
      }
      const parsedArguments = args ? JSON.parse(args) : {};
      const parsedResponse = {
        content: JSON.parse((_response$content = response.content) !== null && _response$content !== void 0 ? _response$content : '{}'),
        data: JSON.parse((_response$data = response.data) !== null && _response$data !== void 0 ? _response$data : '{}')
      };
      return fn === null || fn === void 0 ? void 0 : fn({
        response: parsedResponse,
        arguments: parsedArguments,
        onActionClick
      });
    },
    getFunctions,
    hasFunction: name => {
      return functionRegistry.has(name);
    },
    hasRenderFunction: name => {
      return renderFunctionRegistry.has(name);
    },
    getSystemMessage: () => {
      return {
        '@timestamp': new Date().toISOString(),
        message: {
          role: _common2.MessageRole.System,
          content: systemMessage
        }
      };
    },
    ...client
  };
}