"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerQueryFunction = registerQueryFunction;
var _fs = _interopRequireDefault(require("fs"));
var _lodash = require("lodash");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _path = _interopRequireDefault(require("path"));
var _rxjs = require("rxjs");
var _util = require("util");
var _common = require("@kbn/observability-ai-assistant-plugin/common");
var _visualize_esql = require("@kbn/observability-ai-assistant-plugin/common/functions/visualize_esql");
var _concatenate_chat_completion_chunks = require("@kbn/observability-ai-assistant-plugin/common/utils/concatenate_chat_completion_chunks");
var _emit_with_concatenated_message = require("@kbn/observability-ai-assistant-plugin/common/utils/emit_with_concatenated_message");
var _create_function_response_message = require("@kbn/observability-ai-assistant-plugin/common/utils/create_function_response_message");
var _correct_common_esql_mistakes = require("./correct_common_esql_mistakes");
var _validate_esql_query = require("./validate_esql_query");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const readFile = (0, _util.promisify)(_fs.default.readFile);
const readdir = (0, _util.promisify)(_fs.default.readdir);
const loadSystemMessage = (0, _lodash.once)(async () => {
  const data = await readFile(_path.default.join(__dirname, './system_message.txt'));
  return data.toString('utf-8');
});
const loadEsqlDocs = (0, _lodash.once)(async () => {
  const dir = _path.default.join(__dirname, './esql_docs');
  const files = (await readdir(dir)).filter(file => _path.default.extname(file) === '.txt');
  if (!files.length) {
    return {};
  }
  const limiter = (0, _pLimit.default)(10);
  return (0, _lodash.keyBy)(await Promise.all(files.map(file => limiter(async () => {
    const data = (await readFile(_path.default.join(dir, file))).toString('utf-8');
    const filename = _path.default.basename(file, '.txt');
    const keyword = filename.replace('esql-', '').replace('agg-', '').replaceAll('-', '_').toUpperCase();
    return {
      keyword: keyword === 'STATS_BY' ? 'STATS' : keyword,
      data
    };
  }))), 'keyword');
});
function registerQueryFunction({
  functions,
  resources
}) {
  functions.registerInstruction(({
    availableFunctionNames
  }) => availableFunctionNames.includes('query') ? `You MUST use the "query" function when the user wants to:
  - visualize data
  - run any arbitrary query
  - breakdown or filter ES|QL queries that are displayed on the current page
  - convert queries from another language to ES|QL
  - asks general questions about ES|QL

  DO NOT UNDER ANY CIRCUMSTANCES generate ES|QL queries or explain anything about the ES|QL query language yourself.
  DO NOT UNDER ANY CIRCUMSTANCES try to correct an ES|QL query yourself - always use the "query" function for this.

  Even if the "context" function was used before that, follow it up with the "query" function. If a query fails, do not attempt to correct it yourself. Again you should call the "query" function,
  even if it has been called before.

  When the "visualize_query" function has been called, a visualization has been displayed to the user. DO NOT UNDER ANY CIRCUMSTANCES follow up a "visualize_query" function call with your own visualization attempt.
  If the "execute_query" function has been called, summarize these results for the user. The user does not see a visualization in this case.` : undefined);
  functions.registerFunction({
    name: 'execute_query',
    visibility: _common.FunctionVisibility.UserOnly,
    description: 'Display the results of an ES|QL query. ONLY use this if the "query" function has been used before or if the user or screen context has provided a query you can use.',
    parameters: {
      type: 'object',
      properties: {
        query: {
          type: 'string'
        }
      },
      required: ['query']
    }
  }, async ({
    arguments: {
      query
    }
  }) => {
    const client = (await resources.context.core).elasticsearch.client.asCurrentUser;
    const {
      error,
      errorMessages
    } = await (0, _validate_esql_query.validateEsqlQuery)({
      query,
      client
    });
    if (!!error) {
      return {
        content: {
          message: 'The query failed to execute',
          error,
          errorMessages
        }
      };
    }
    const response = await client.transport.request({
      method: 'POST',
      path: '_query',
      body: {
        query
      }
    });
    return {
      content: response
    };
  });
  functions.registerFunction({
    name: 'query',
    description: `This function generates, executes and/or visualizes a query based on the user's request. It also explains how ES|QL works and how to convert queries from one language to another. Make sure you call one of the get_dataset functions first if you need index or field names. This function takes no input.`,
    visibility: _common.FunctionVisibility.AssistantOnly
  }, async ({
    messages,
    chat
  }, signal) => {
    var _args$commands, _args$functions;
    const [systemMessage, esqlDocs] = await Promise.all([loadSystemMessage(), loadEsqlDocs()]);
    const withEsqlSystemMessage = message => [{
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common.MessageRole.System,
        content: `${systemMessage}\n${message !== null && message !== void 0 ? message : ''}`
      }
    }, ...messages.slice(1)];
    const source$ = (await chat('classify_esql', {
      messages: withEsqlSystemMessage().concat({
        '@timestamp': new Date().toISOString(),
        message: {
          role: _common.MessageRole.User,
          content: `Use the classify_esql function to classify the user's request
              in the user message before this.
              and get more information about specific functions and commands
              you think are candidates for answering the question.
              
              Examples for functions and commands:
              Do you need to group data? Request \`STATS\`.
              Extract data? Request \`DISSECT\` AND \`GROK\`.
              Convert a column based on a set of conditionals? Request \`EVAL\` and \`CASE\`.

              ONLY use ${_visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults} if you are absolutely sure
              it is executable. If one of the get_dataset_info functions were not called before, OR if
              one of the get_dataset_info functions returned no data, opt for an explanation only and
              mention that there is no data for these indices. You can still use
              ${_visualize_esql.VisualizeESQLUserIntention.generateQueryOnly} and generate an example ES|QL query.

              For determining the intention of the user, the following options are available:

              ${_visualize_esql.VisualizeESQLUserIntention.generateQueryOnly}: the user only wants to generate the query,
              but not run it, or they ask a general question about ES|QL.

              ${_visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults}: the user wants to execute the query,
              and have the assistant return/analyze/summarize the results. they don't need a
              visualization.

              ${_visualize_esql.VisualizeESQLUserIntention.visualizeAuto}: The user wants to visualize the data from the
              query, but wants us to pick the best visualization type, or their preferred
              visualization is unclear.

              These intentions will display a specific visualization:
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeBar}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeDonut}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeHeatmap}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeLine}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeArea}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeTable}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeTagcloud}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeTreemap}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeWaffle}
              ${_visualize_esql.VisualizeESQLUserIntention.visualizeXy}

              Some examples:

              "I want a query that ..." => ${_visualize_esql.VisualizeESQLUserIntention.generateQueryOnly}
              "... Just show me the query" => ${_visualize_esql.VisualizeESQLUserIntention.generateQueryOnly}
              "Create a query that ..." => ${_visualize_esql.VisualizeESQLUserIntention.generateQueryOnly}
              
              "Show me the avg of x" => ${_visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults}
              "Show me the results of y" => ${_visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults}
              "Display the sum of z" => ${_visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults}

              "Show me the avg of x over time" => ${_visualize_esql.VisualizeESQLUserIntention.visualizeAuto}
              "I want a bar chart of ... " => ${_visualize_esql.VisualizeESQLUserIntention.visualizeBar}
              "I want to see a heat map of ..." => ${_visualize_esql.VisualizeESQLUserIntention.visualizeHeatmap}
              `
        }
      }),
      signal,
      functions: [{
        name: 'classify_esql',
        description: `Use this function to determine:
              - what ES|QL functions and commands are candidates for answering the user's question
              - whether the user has requested a query, and if so, it they want it to be executed, or just shown.

              All parameters are required. Make sure the functions and commands you request are available in the
              system message.
              `,
        parameters: {
          type: 'object',
          properties: {
            commands: {
              type: 'array',
              items: {
                type: 'string'
              },
              description: 'A list of processing or source commands'
            },
            functions: {
              type: 'array',
              items: {
                type: 'string'
              },
              description: 'A list of functions.'
            },
            intention: {
              type: 'string',
              description: `What the user\'s intention is.`,
              enum: _visualize_esql.VISUALIZE_ESQL_USER_INTENTIONS
            }
          },
          required: ['commands', 'functions', 'intention']
        }
      }],
      functionCall: 'classify_esql'
    })).pipe((0, _concatenate_chat_completion_chunks.concatenateChatCompletionChunks)());
    const response = await (0, _rxjs.lastValueFrom)(source$);
    if (!response.message.function_call.arguments) {
      throw new Error('LLM did not call classify_esql function');
    }
    const args = JSON.parse(response.message.function_call.arguments);
    const keywords = [...((_args$commands = args.commands) !== null && _args$commands !== void 0 ? _args$commands : []), ...((_args$functions = args.functions) !== null && _args$functions !== void 0 ? _args$functions : []), 'SYNTAX', 'OVERVIEW', 'OPERATORS'].map(keyword => keyword.toUpperCase());
    const messagesToInclude = (0, _lodash.mapValues)((0, _lodash.pick)(esqlDocs, keywords), ({
      data
    }) => data);
    let userIntentionMessage;
    switch (args.intention) {
      case _visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults:
        userIntentionMessage = `When you generate a query, it will automatically be executed and its results returned to you. The user does not need to do anything for this.`;
        break;
      case _visualize_esql.VisualizeESQLUserIntention.generateQueryOnly:
        userIntentionMessage = `Any generated query will not be executed automatically, the user needs to do this themselves.`;
        break;
      default:
        userIntentionMessage = `The generated query will automatically be visualized to the user, displayed below your message. The user does not need to do anything for this.`;
        break;
    }
    const esqlResponse$ = await chat('answer_esql_question', {
      messages: [...withEsqlSystemMessage(), {
        '@timestamp': new Date().toISOString(),
        message: {
          role: _common.MessageRole.Assistant,
          content: '',
          function_call: {
            name: 'get_esql_info',
            arguments: JSON.stringify(args),
            trigger: _common.MessageRole.Assistant
          }
        }
      }, {
        '@timestamp': new Date().toISOString(),
        message: {
          role: _common.MessageRole.User,
          name: 'get_esql_info',
          content: JSON.stringify({
            documentation: messagesToInclude
          })
        }
      }, {
        '@timestamp': new Date().toISOString(),
        message: {
          role: _common.MessageRole.User,
          content: `Answer the user's question that was previously asked using the attached documentation.

                Format any ES|QL query as follows:
                \`\`\`esql
                <query>
                \`\`\`

                Respond in plain text. Do not attempt to use a function.
  
                Prefer to use commands and functions for which you have requested documentation.
  
                ${args.intention !== _visualize_esql.VisualizeESQLUserIntention.generateQueryOnly ? `DO NOT UNDER ANY CIRCUMSTANCES generate more than a single query.
                    If multiple queries are needed, do it as a follow-up step. Make this clear to the user. For example:
                    
                    Human: plot both yesterday's and today's data.
                    
                    Assistant: Here's how you can plot yesterday's data:
                    \`\`\`esql
                    <query>
                    \`\`\`
  
                    Let's see that first. We'll look at today's data next.
  
                    Human: <response from yesterday's data>
  
                    Assistant: Let's look at today's data:
  
                    \`\`\`esql
                    <query>
                    \`\`\`
                    ` : ''}
  
                ${userIntentionMessage}
  
                DO NOT UNDER ANY CIRCUMSTANCES use commands or functions that are not a capability of ES|QL
                as mentioned in the system message and documentation. When converting queries from one language
                to ES|QL, make sure that the functions are available and documented in ES|QL.
                E.g., for SPL's LEN, use LENGTH. For IF, use CASE.
                
                `
        }
      }],
      signal,
      functions: functions.getActions()
    });
    return esqlResponse$.pipe((0, _emit_with_concatenated_message.emitWithConcatenatedMessage)(async msg => {
      var _msg$message$content$;
      msg.message.content = msg.message.content.replaceAll(_constants.INLINE_ESQL_QUERY_REGEX, (_match, query) => {
        const correction = (0, _correct_common_esql_mistakes.correctCommonEsqlMistakes)(query);
        if (correction.isCorrection) {
          resources.logger.debug(`Corrected query, from: \n${correction.input}\nto:\n${correction.output}`);
        }
        return '```esql\n' + correction.output + '\n```';
      });
      if (msg.message.function_call.name) {
        return msg;
      }
      const esqlQuery = (_msg$message$content$ = msg.message.content.match(new RegExp(_constants.INLINE_ESQL_QUERY_REGEX, 'ms'))) === null || _msg$message$content$ === void 0 ? void 0 : _msg$message$content$[1];
      let functionCall;
      if (!args.intention || !esqlQuery || args.intention === _visualize_esql.VisualizeESQLUserIntention.generateQueryOnly) {
        functionCall = undefined;
      } else if (args.intention === _visualize_esql.VisualizeESQLUserIntention.executeAndReturnResults) {
        functionCall = {
          name: 'execute_query',
          arguments: JSON.stringify({
            query: esqlQuery
          }),
          trigger: _common.MessageRole.Assistant
        };
      } else {
        functionCall = {
          name: 'visualize_query',
          arguments: JSON.stringify({
            query: esqlQuery,
            intention: args.intention
          }),
          trigger: _common.MessageRole.Assistant
        };
      }
      return {
        ...msg,
        message: {
          ...msg.message,
          ...(functionCall ? {
            function_call: functionCall
          } : {})
        }
      };
    }), (0, _rxjs.startWith)((0, _create_function_response_message.createFunctionResponseMessage)({
      name: 'query',
      content: {},
      data: {
        // add the included docs for debugging
        documentation: {
          intention: args.intention,
          keywords,
          files: messagesToInclude
        }
      }
    })));
  });
}