"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QuickCategorizationJobCreator = exports.CATEGORIZATION_TYPE = void 0;
var _i18n = require("@kbn/i18n");
var _mlAnomalyUtils = require("@kbn/ml-anomaly-utils");
var _new_job = require("../../../../../common/constants/new_job");
var _quick_create_job_base = require("../job_from_dashboard/quick_create_job_base");
var _default_configs = require("../common/job_creator/util/default_configs");
var _general = require("../common/job_creator/util/general");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CATEGORIZATION_TYPE = exports.CATEGORIZATION_TYPE = {
  COUNT: _mlAnomalyUtils.ML_JOB_AGGREGATION.COUNT,
  HIGH_COUNT: _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_COUNT,
  RARE: _mlAnomalyUtils.ML_JOB_AGGREGATION.RARE
};
class QuickCategorizationJobCreator extends _quick_create_job_base.QuickJobCreatorBase {
  constructor(dataViews, kibanaConfig, timeFilter, dashboardService, data, mlApiServices) {
    super(dataViews, kibanaConfig, timeFilter, dashboardService, mlApiServices);
    this.data = data;
  }
  async createAndSaveJob(categorizationType, jobId, bucketSpan, dataView, field, partitionField, stopOnWarn, query, timeRange, startJob, runInRealTime) {
    if (query === undefined) {
      throw new Error('Cannot create job, query and filters are undefined');
    }
    const {
      jobConfig,
      datafeedConfig,
      start,
      end
    } = await this.createJob(categorizationType, dataView, field, partitionField, stopOnWarn, timeRange, query, bucketSpan);
    const createdByLabel = _new_job.CREATED_BY_LABEL.CATEGORIZATION_FROM_PATTERN_ANALYSIS;
    const result = await this.putJobAndDataFeed({
      jobId,
      datafeedConfig,
      jobConfig,
      createdByLabel,
      start,
      end,
      startJob,
      runInRealTime
    });
    return result;
  }
  async createAndStashADJob(categorizationType, dataViewId, fieldName, partitionFieldName, stopOnWarn, startString, endString, query) {
    try {
      var _dataView$getFieldByN;
      const dataView = await this.data.dataViews.get(dataViewId);
      const field = dataView.getFieldByName(fieldName);
      const partitionField = partitionFieldName ? (_dataView$getFieldByN = dataView.getFieldByName(partitionFieldName)) !== null && _dataView$getFieldByN !== void 0 ? _dataView$getFieldByN : null : null;
      if (field === undefined) {
        throw new Error('Cannot create job, field is undefined');
      }
      const {
        jobConfig,
        datafeedConfig,
        start,
        end,
        includeTimeRange
      } = await this.createJob(categorizationType, dataView, field, partitionField, stopOnWarn, {
        from: startString,
        to: endString
      }, query, _new_job.DEFAULT_BUCKET_SPAN);

      // add job config and start and end dates to the
      // job cloning stash, so they can be used
      // by the new job wizards
      (0, _general.stashJobForCloning)({
        jobConfig,
        datafeedConfig,
        createdBy: _new_job.CREATED_BY_LABEL.CATEGORIZATION_FROM_PATTERN_ANALYSIS,
        start,
        end
      }, true, includeTimeRange, !includeTimeRange);
    } catch (error) {
      // eslint-disable-next-line no-console
      console.error(error);
    }
  }
  async createJob(categorizationType, dataView, field, partitionField, stopOnWarn, timeRange, query, bucketSpan) {
    const jobConfig = (0, _default_configs.createEmptyJob)();
    const datafeedConfig = (0, _default_configs.createEmptyDatafeed)(dataView.getIndexPattern());
    datafeedConfig.query = query;
    jobConfig.analysis_config = {
      categorization_field_name: field.name,
      per_partition_categorization: {
        enabled: partitionField !== null,
        stop_on_warn: stopOnWarn
      },
      influencers: [_mlAnomalyUtils.MLCATEGORY],
      detectors: [{
        function: categorizationType,
        by_field_name: _mlAnomalyUtils.MLCATEGORY
      }],
      bucket_span: bucketSpan
    };
    if (partitionField !== null) {
      jobConfig.analysis_config.detectors[0].partition_field_name = partitionField.name;
      jobConfig.analysis_config.influencers.push(partitionField.name);
    }
    jobConfig.data_description.time_field = dataView.timeFieldName;
    let start;
    let end;
    let includeTimeRange = true;
    try {
      // attempt to parse the start and end dates.
      // if start and end values cannot be determined
      // instruct the job cloning code to auto-select the
      // full time range for the index.
      const {
        min,
        max
      } = this.timeFilter.calculateBounds(timeRange);
      start = min === null || min === void 0 ? void 0 : min.valueOf();
      end = max === null || max === void 0 ? void 0 : max.valueOf();
      if (start === undefined || end === undefined || isNaN(start) || isNaN(end)) {
        throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.timeRange', {
          defaultMessage: 'Incompatible time range'
        }));
      }
    } catch (error) {
      // eslint-disable-next-line no-console
      console.error(error);
      includeTimeRange = false;
      start = undefined;
      end = undefined;
    }
    return {
      jobConfig,
      datafeedConfig,
      start,
      end,
      includeTimeRange
    };
  }
}
exports.QuickCategorizationJobCreator = QuickCategorizationJobCreator;