"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.downloadCsvShareProvider = void 0;
var _i18n = require("@kbn/i18n");
var _react = _interopRequireDefault(require("react"));
var _common = require("@kbn/data-plugin/common");
var _public = require("@kbn/share-plugin/public");
var _public2 = require("@kbn/data-plugin/public");
var _i18nReact = require("@kbn/i18n-react");
var _csv_download_panel_content_lazy = require("./csv_download_panel_content_lazy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function downloadCSVs({
  activeData,
  title,
  formatFactory,
  uiSettings,
  columnsSorting
}) {
  if (!activeData) {
    if (window.ELASTIC_LENS_CSV_DOWNLOAD_DEBUG) {
      window.ELASTIC_LENS_CSV_CONTENT = undefined;
    }
    return;
  }
  const datatables = Object.values(activeData);
  const content = datatables.reduce((memo, datatable, i) => {
    // skip empty datatables
    if (datatable) {
      const postFix = datatables.length > 1 ? `-${i + 1}` : '';
      memo[`${title}${postFix}.csv`] = {
        content: _public2.exporters.datatableToCSV(datatable, {
          csvSeparator: uiSettings.get('csv:separator', ','),
          quoteValues: uiSettings.get('csv:quoteValues', true),
          formatFactory,
          escapeFormulaValues: false,
          columnsSorting
        }),
        type: _public2.exporters.CSV_MIME_TYPE
      };
    }
    return memo;
  }, {});
  if (window.ELASTIC_LENS_CSV_DOWNLOAD_DEBUG) {
    window.ELASTIC_LENS_CSV_CONTENT = content;
  }
  if (content) {
    (0, _public.downloadMultipleAs)(content);
  }
}
function getWarnings(activeData) {
  const messages = [];
  if (activeData) {
    const datatables = Object.values(activeData);
    const formulaDetected = datatables.some(datatable => {
      return (0, _common.tableHasFormulas)(datatable.columns, datatable.rows);
    });
    if (formulaDetected) {
      messages.push(_i18n.i18n.translate('xpack.lens.app.downloadButtonFormulasWarning', {
        defaultMessage: 'Your CSV contains characters that spreadsheet applications might interpret as formulas.'
      }));
    }
  }
  return messages;
}
const downloadCsvShareProvider = ({
  uiSettings,
  formatFactoryFn,
  atLeastGold,
  isNewVersion
}) => {
  const getShareMenuItems = ({
    objectType,
    sharingData
  }) => {
    if ('lens' !== objectType) {
      return [];
    }
    const {
      title,
      activeData,
      csvEnabled,
      columnsSorting
    } = sharingData;
    const panelTitle = _i18n.i18n.translate('xpack.lens.reporting.shareContextMenu.csvReportsButtonLabel', {
      defaultMessage: 'CSV Download'
    });
    const menuItemMetadata = {
      shareMenuItem: {
        name: panelTitle,
        icon: 'document',
        disabled: !csvEnabled,
        sortOrder: 1
      }
    };
    const downloadCSVHandler = () => downloadCSVs({
      title,
      formatFactory: formatFactoryFn(),
      activeData,
      uiSettings,
      columnsSorting
    });
    if (!isNewVersion) {
      return [{
        ...menuItemMetadata,
        panel: {
          id: 'csvDownloadPanel',
          title: panelTitle,
          content: /*#__PURE__*/_react.default.createElement(_csv_download_panel_content_lazy.DownloadPanelContent, {
            isDisabled: !csvEnabled,
            warnings: getWarnings(activeData),
            onClick: downloadCSVHandler
          })
        }
      }];
    }
    return [{
      ...menuItemMetadata,
      label: 'CSV',
      reportType: 'lens_csv',
      downloadCSVLens: downloadCSVHandler,
      ...(atLeastGold() ? {
        helpText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.lens.share.helpText",
          defaultMessage: "Select the file type you would like to export for this visualization."
        }),
        generateReportButton: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.lens.share.export",
          defaultMessage: "Generate export"
        }),
        renderLayoutOptionSwitch: false,
        getJobParams: undefined,
        showRadios: true
      } : {
        isDisabled: !csvEnabled,
        warnings: getWarnings(activeData),
        helpText: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.lens.application.csvPanelContent.generationDescription",
          defaultMessage: "Download the data displayed in the visualization."
        }),
        generateReportButton: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.lens.share.csvButton",
          "data-test-subj": "generateReportButton",
          defaultMessage: "Download CSV"
        })
      })
    }];
  };
  return {
    id: 'csvDownloadLens',
    getShareMenuItems
  };
};
exports.downloadCsvShareProvider = downloadCsvShareProvider;