"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetAllRoute = registerGetAllRoute;
exports.registerGetOneRoute = registerGetOneRoute;
var _configSchema = require("@kbn/config-schema");
var _lib = require("../../../../common/lib");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const enhanceDataStreams = ({
  dataStreams,
  dataStreamsStats,
  dataStreamsPrivileges
}) => {
  return dataStreams.map(dataStream => {
    const enhancedDataStream = {
      ...dataStream,
      privileges: {
        delete_index: dataStreamsPrivileges ? dataStreamsPrivileges.index[dataStream.name].delete_index : true,
        manage_data_stream_lifecycle: dataStreamsPrivileges ? dataStreamsPrivileges.index[dataStream.name].manage_data_stream_lifecycle : true
      }
    };
    if (dataStreamsStats) {
      const currentDataStreamStats = dataStreamsStats.find(({
        data_stream: statsName
      }) => statsName === dataStream.name);
      if (currentDataStreamStats) {
        enhancedDataStream.store_size = currentDataStreamStats.store_size;
        enhancedDataStream.store_size_bytes = currentDataStreamStats.store_size_bytes;
        enhancedDataStream.maximum_timestamp = currentDataStreamStats.maximum_timestamp;
      }
    }
    return enhancedDataStream;
  });
};
const getDataStreams = (client, name = '*') => {
  return client.asCurrentUser.indices.getDataStream({
    name,
    expand_wildcards: 'all'
  });
};
const getDataStreamsStats = (client, name = '*') => {
  return client.asCurrentUser.indices.dataStreamsStats({
    name,
    expand_wildcards: 'all',
    human: true
  });
};
const getDataStreamsPrivileges = (client, names) => {
  return client.asCurrentUser.security.hasPrivileges({
    body: {
      index: [{
        names,
        privileges: ['delete_index', 'manage_data_stream_lifecycle']
      }]
    }
  });
};
function registerGetAllRoute({
  router,
  lib: {
    handleEsError
  },
  config
}) {
  const querySchema = _configSchema.schema.object({
    includeStats: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('true'), _configSchema.schema.literal('false')]))
  });
  router.get({
    path: (0, _.addBasePath)('/data_streams'),
    validate: {
      query: querySchema
    }
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const includeStats = request.query.includeStats === 'true';
    try {
      const {
        data_streams: dataStreams
      } = await getDataStreams(client);
      let dataStreamsStats;
      let dataStreamsPrivileges;
      if (includeStats) {
        ({
          data_streams: dataStreamsStats
        } = await getDataStreamsStats(client));
      }
      if (config.isSecurityEnabled() && dataStreams.length > 0) {
        dataStreamsPrivileges = await getDataStreamsPrivileges(client, dataStreams.map(dataStream => dataStream.name));
      }
      const enhancedDataStreams = enhanceDataStreams({
        dataStreams,
        dataStreamsStats,
        dataStreamsPrivileges
      });
      return response.ok({
        body: (0, _lib.deserializeDataStreamList)(enhancedDataStreams)
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}
function registerGetOneRoute({
  router,
  lib: {
    handleEsError
  },
  config
}) {
  const paramsSchema = _configSchema.schema.object({
    name: _configSchema.schema.string()
  });
  router.get({
    path: (0, _.addBasePath)('/data_streams/{name}'),
    validate: {
      params: paramsSchema
    }
  }, async (context, request, response) => {
    const {
      name
    } = request.params;
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const [{
        data_streams: dataStreams
      }, {
        data_streams: dataStreamsStats
      }] = await Promise.all([getDataStreams(client, name), getDataStreamsStats(client, name)]);
      if (dataStreams[0]) {
        let dataStreamsPrivileges;
        if (config.isSecurityEnabled()) {
          dataStreamsPrivileges = await getDataStreamsPrivileges(client, [dataStreams[0].name]);
        }
        const enhancedDataStreams = enhanceDataStreams({
          dataStreams,
          dataStreamsStats,
          dataStreamsPrivileges
        });
        const body = (0, _lib.deserializeDataStream)(enhancedDataStreams[0]);
        return response.ok({
          body
        });
      }
      return response.notFound();
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}