"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchCursorScroll = void 0;
var _rxjs = require("rxjs");
var _common = require("@kbn/data-plugin/common");
var _search_cursor = require("./search_cursor");
var _i18n_texts = require("./i18n_texts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class SearchCursorScroll extends _search_cursor.SearchCursor {
  constructor(indexPatternTitle, settings, clients, abortController, logger) {
    super(indexPatternTitle, settings, clients, abortController, logger);
  }

  // The first search query begins the scroll context in ES
  async initialize() {}
  async scan(searchBody) {
    const {
      includeFrozen,
      maxConcurrentShardRequests,
      scroll,
      taskInstanceFields
    } = this.settings;

    // maxConcurrentShardRequests=0 is not supported
    const effectiveMaxConcurrentShardRequests = maxConcurrentShardRequests > 0 ? maxConcurrentShardRequests : undefined;
    const searchParamsScan = {
      params: {
        body: searchBody,
        index: this.indexPatternTitle,
        scroll: scroll.duration(taskInstanceFields),
        size: scroll.size,
        ignore_throttled: includeFrozen ? false : undefined,
        // "true" will cause deprecation warnings logged in ES
        max_concurrent_shard_requests: effectiveMaxConcurrentShardRequests
      }
    };
    return await (0, _rxjs.lastValueFrom)(this.clients.data.search(searchParamsScan, {
      strategy: _common.ES_SEARCH_STRATEGY,
      abortSignal: this.abortController.signal,
      transport: {
        maxRetries: 0,
        // retrying reporting jobs is handled in the task manager scheduling logic
        requestTimeout: scroll.duration(taskInstanceFields)
      }
    }));
  }
  async scroll() {
    const {
      scroll,
      taskInstanceFields
    } = this.settings;
    return await this.clients.es.asCurrentUser.scroll({
      scroll: scroll.duration(taskInstanceFields),
      scroll_id: this.cursorId
    }, {
      signal: this.abortController.signal,
      maxRetries: 0,
      // retrying reporting jobs is handled in the task manager scheduling logic
      requestTimeout: scroll.duration(taskInstanceFields)
    });
  }
  async getPage(searchSource) {
    if (this.cursorId) {
      this.logger.debug(`Executing search request with scroll ID [${this.formatCursorId(this.cursorId)}]`);
    } else {
      this.logger.debug(`Executing search for initial scan request.`);
    }
    let response;
    const searchBody = searchSource.getSearchRequestBody();
    if (searchBody == null) {
      throw new Error('Could not retrieve the search body!');
    }
    if (this.cursorId == null) {
      response = await this.scan(searchBody);
    } else {
      response = {
        rawResponse: await this.scroll()
      };
    }
    if (!response) {
      throw new Error(`Response could not be retrieved!`);
    }
    const {
      rawResponse,
      ...rawDetails
    } = response;
    this.logSearchResults(rawDetails, rawResponse);
    this.logger.debug(`Received Scroll ID: [${this.formatCursorId(rawResponse._scroll_id)}]`);
    return rawResponse;
  }
  updateIdFromResults(results) {
    var _results$_scroll_id;
    this.cursorId = (_results$_scroll_id = results._scroll_id) !== null && _results$_scroll_id !== void 0 ? _results$_scroll_id : this.cursorId;
  }
  async closeCursor() {
    if (this.cursorId) {
      this.logger.debug(`Executing clearScroll on ${this.formatCursorId(this.cursorId)}`);
      await this.clients.es.asCurrentUser.clearScroll({
        scroll_id: [this.cursorId]
      });
    } else {
      this.logger.warn(`No Scroll Id to clear!`);
    }
  }
  getUnableToCloseCursorMessage() {
    return _i18n_texts.i18nTexts.csvUnableToCloseScroll();
  }
}
exports.SearchCursorScroll = SearchCursorScroll;