"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkFleetServerVersionsForSecretsStorage = checkFleetServerVersionsForSecretsStorage;
exports.hasFleetServers = hasFleetServers;
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _coerce = _interopRequireDefault(require("semver/functions/coerce"));
var _constants = require("../../constants");
var _agents = require("../agents");
var _package_policy = require("../package_policy");
var _agent_policy = require("../agent_policy");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Check if at least one fleet server is connected
 */
async function hasFleetServers(esClient) {
  const res = await esClient.search({
    index: _constants.FLEET_SERVER_SERVERS_INDEX,
    ignore_unavailable: true,
    filter_path: 'hits.total',
    track_total_hits: true,
    rest_total_hits_as_int: true
  });
  return res.hits.total > 0;
}

/**
 * This function checks if all Fleet Server agents are running at least a given version, but with
 * some important caveats related to enabling the secrets storage feature:
 *
 * 1. Any unenrolled agents are ignored if they are running an outdated version
 * 2. Managed agents in an inactive state are ignored if they are running an outdated version.
 */
async function checkFleetServerVersionsForSecretsStorage(esClient, soClient, version) {
  const logger = _.appContextService.getLogger();
  let hasMore = true;
  const policyIds = new Set();
  let page = 1;
  while (hasMore) {
    const res = await _package_policy.packagePolicyService.list(soClient, {
      page: page++,
      perPage: 20,
      kuery: 'ingest-package-policies.package.name:fleet_server'
    });
    for (const item of res.items) {
      policyIds.add(item.policy_id);
    }
    if (res.items.length === 0) {
      hasMore = false;
    }
  }
  const managedAgentPolicies = await _agent_policy.agentPolicyService.getAllManagedAgentPolicies(soClient);
  const fleetServerAgents = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
    showInactive: true,
    perPage: _constants.SO_SEARCH_LIMIT
  });
  if (fleetServerAgents.agents.length === 0) {
    return false;
  }
  let result = true;
  for (const fleetServerAgent of fleetServerAgents.agents) {
    var _fleetServerAgent$loc, _fleetServerAgent$loc2, _fleetServerAgent$loc3;
    const agentVersion = (_fleetServerAgent$loc = fleetServerAgent.local_metadata) === null || _fleetServerAgent$loc === void 0 ? void 0 : (_fleetServerAgent$loc2 = _fleetServerAgent$loc.elastic) === null || _fleetServerAgent$loc2 === void 0 ? void 0 : (_fleetServerAgent$loc3 = _fleetServerAgent$loc2.agent) === null || _fleetServerAgent$loc3 === void 0 ? void 0 : _fleetServerAgent$loc3.version;
    if (!agentVersion) {
      continue;
    }
    const isNewerVersion = (0, _gte.default)((0, _coerce.default)(agentVersion), version);
    if (!isNewerVersion) {
      const agentStatus = await (0, _agents.getAgentStatusById)(esClient, soClient, fleetServerAgent.id);

      // Any unenrolled Fleet Server agents can be ignored
      if (agentStatus === 'unenrolled' || fleetServerAgent.status === 'unenrolling' || fleetServerAgent.unenrolled_at) {
        logger.debug(`Found unenrolled Fleet Server agent ${fleetServerAgent.id} on version ${agentVersion} when checking for secrets storage compatibility - ignoring`);
        continue;
      }
      const isManagedAgentPolicy = managedAgentPolicies.some(managedPolicy => managedPolicy.id === fleetServerAgent.policy_id);

      // If this is an agent enrolled in a managed policy, and it is no longer active then we ignore it if it's
      // running on an outdated version. This prevents users with offline Elastic Agent on Cloud policies from
      // being stuck when it comes to enabling secrets, as agents can't be unenrolled from managed policies via Fleet UI.
      if (isManagedAgentPolicy && ['offline', 'inactive'].includes(agentStatus) || !fleetServerAgent.active) {
        logger.debug(`Found outdated managed Fleet Server agent ${fleetServerAgent.id} on version ${agentVersion} when checking for secrets storage compatibility - ignoring due to ${agentStatus} status`);
        continue;
      }
      logger.debug(`Found outdated Fleet Server agent ${fleetServerAgent.id} on version ${agentVersion} - secrets won't be enabled until all Fleet Server agents are running at least ${version}`);
      result = false;
      break;
    }
  }
  return result;
}