"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActions = getActions;
exports.getCompatibleFunctionDefinitions = void 0;
exports.getMetaFieldsRetriever = getMetaFieldsRetriever;
var _i18n = require("@kbn/i18n");
var _jsLevenshtein = _interopRequireDefault(require("js-levenshtein"));
var _lodash = require("lodash");
var _resources_helpers = require("../shared/resources_helpers");
var _helpers = require("../shared/helpers");
var _helpers2 = require("../validation/helpers");
var _constants = require("../shared/constants");
var _context = require("../shared/context");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getFieldsByTypeRetriever(queryString, resourceRetriever) {
  const helpers = (0, _resources_helpers.getFieldsByTypeHelper)(queryString, resourceRetriever);
  return {
    getFieldsByType: async (expectedType = 'any', ignored = []) => {
      const fields = await helpers.getFieldsByType(expectedType, ignored);
      return fields;
    },
    getFieldsMap: helpers.getFieldsMap
  };
}
function getPolicyRetriever(resourceRetriever) {
  const helpers = (0, _resources_helpers.getPolicyHelper)(resourceRetriever);
  return {
    getPolicies: async () => {
      const policies = await helpers.getPolicies();
      return policies.map(({
        name
      }) => name);
    },
    getPolicyFields: async policy => {
      const metadata = await helpers.getPolicyMetadata(policy);
      return (metadata === null || metadata === void 0 ? void 0 : metadata.enrichFields) || [];
    }
  };
}
function getSourcesRetriever(resourceRetriever) {
  const helper = (0, _resources_helpers.getSourcesHelper)(resourceRetriever);
  return async () => {
    const list = (await helper()) || [];
    // hide indexes that start with .
    return list.filter(({
      hidden
    }) => !hidden).map(({
      name
    }) => name);
  };
}
function getMetaFieldsRetriever(queryString, commands, callbacks) {
  return async () => {
    if (!callbacks || !callbacks.getMetaFields) {
      return [];
    }
    return await callbacks.getMetaFields();
  };
}
const getCompatibleFunctionDefinitions = (command, option) => {
  const fnSupportedByCommand = (0, _helpers.getAllFunctions)({
    type: ['eval', 'agg']
  }).filter(({
    name,
    supportedCommands,
    supportedOptions
  }) => option ? supportedOptions === null || supportedOptions === void 0 ? void 0 : supportedOptions.includes(option) : supportedCommands.includes(command));
  return fnSupportedByCommand.map(({
    name
  }) => name);
};
exports.getCompatibleFunctionDefinitions = getCompatibleFunctionDefinitions;
function createAction(title, solution, error) {
  return {
    title,
    diagnostics: [error],
    kind: 'quickfix',
    edits: [{
      range: error,
      text: solution
    }]
  };
}
async function getSpellingPossibilities(fn, errorText) {
  const allPossibilities = await fn();
  const allSolutions = allPossibilities.reduce((solutions, item) => {
    const distance = (0, _jsLevenshtein.default)(item, errorText);
    if (distance < 3) {
      solutions.push(item);
    }
    return solutions;
  }, []);
  // filter duplicates
  return Array.from(new Set(allSolutions));
}
async function getSpellingActionForColumns(error, queryString, ast, options, {
  getFieldsByType,
  getPolicies,
  getPolicyFields
}) {
  const errorText = queryString.substring(error.startColumn - 1, error.endColumn - 1);
  if (!getFieldsByType || !getPolicyFields) {
    return [];
  }
  // @TODO add variables support
  const possibleFields = await getSpellingPossibilities(async () => {
    const availableFields = await getFieldsByType('any');
    const enrichPolicies = ast.filter(({
      name
    }) => name === 'enrich');
    if (enrichPolicies.length) {
      const enrichPolicyNames = enrichPolicies.flatMap(({
        args
      }) => args.filter(_helpers.isSourceItem).map(({
        name
      }) => name));
      const enrichFields = await Promise.all(enrichPolicyNames.map(getPolicyFields));
      availableFields.push(...enrichFields.flat());
    }
    return availableFields;
  }, errorText);
  return wrapIntoSpellingChangeAction(error, possibleFields);
}
function extractUnquotedFieldText(query, errorType, ast, possibleStart, end) {
  if (errorType === 'syntaxError') {
    // scope it down to column items for now
    const {
      node
    } = (0, _context.getAstContext)(query, ast, possibleStart - 1);
    if (node && (0, _helpers.isColumnItem)(node)) {
      return {
        start: node.location.min + 1,
        name: query.substring(node.location.min, end).trimEnd()
      };
    }
  }
  return {
    start: possibleStart + 1,
    name: query.substring(possibleStart, end - 1).trimEnd()
  };
}
async function getQuotableActionForColumns(error, queryString, ast, options, {
  getFieldsByType
}) {
  var _ast$find;
  const commandEndIndex = (_ast$find = ast.find(command => error.startColumn > command.location.min && error.startColumn < command.location.max)) === null || _ast$find === void 0 ? void 0 : _ast$find.location.max;

  // the error received is unknwonColumn here, but look around the column to see if there's more
  // which broke the grammar and the validation code couldn't identify as unquoted column
  const remainingCommandText = queryString.substring(error.endColumn - 1, commandEndIndex ? commandEndIndex + 1 : undefined);
  const stopIndex = Math.max(/[()]/.test(remainingCommandText) ? remainingCommandText.indexOf(')') : /,/.test(remainingCommandText) ? remainingCommandText.indexOf(',') - 1 : /\s/.test(remainingCommandText) ? remainingCommandText.indexOf(' ') : remainingCommandText.length, 0);
  const possibleUnquotedText = queryString.substring(error.endColumn - 1, error.endColumn + stopIndex);
  const {
    start,
    name: errorText
  } = extractUnquotedFieldText(queryString, error.code || 'syntaxError', ast, error.startColumn - 1, error.endColumn + possibleUnquotedText.length - 1);
  const actions = [];
  if ((0, _helpers.shouldBeQuotedText)(errorText)) {
    const solution = `\`${errorText.replace(_constants.SINGLE_TICK_REGEX, _constants.DOUBLE_BACKTICK)}\``;
    if (!getFieldsByType) {
      if (!options.relaxOnMissingCallbacks) {
        return [];
      }
      const textHasAlreadyQuotes = /`/.test(errorText);
      if (textHasAlreadyQuotes) {
        return [];
      }
      actions.push(createAction(_i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.quickfix.replaceWithSolution', {
        defaultMessage: 'Did you mean {solution} ?',
        values: {
          solution
        }
      }), solution, {
        ...error,
        startColumn: start,
        endColumn: start + errorText.length
      } // override the location
      ));
    } else {
      const availableFields = new Set(await getFieldsByType('any'));
      if (availableFields.has(errorText) || availableFields.has(solution)) {
        actions.push(createAction(_i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.quickfix.replaceWithSolution', {
          defaultMessage: 'Did you mean {solution} ?',
          values: {
            solution
          }
        }), solution, {
          ...error,
          startColumn: start,
          endColumn: start + errorText.length
        } // override the location
        ));
      }
    }
  }
  return actions;
}
async function getSpellingActionForIndex(error, queryString, ast, options, {
  getSources
}) {
  if (!getSources) {
    return [];
  }
  const errorText = queryString.substring(error.startColumn - 1, error.endColumn - 1);
  const possibleSources = await getSpellingPossibilities(async () => {
    // Handle fuzzy names via truncation to test levenstein distance
    const sources = await getSources();
    if (errorText.endsWith('*')) {
      return sources.map(source => source.length > errorText.length ? source.substring(0, errorText.length - 1) + '*' : source);
    }
    return sources;
  }, errorText);
  return wrapIntoSpellingChangeAction(error, possibleSources);
}
async function getSpellingActionForPolicies(error, queryString, ast, options, {
  getPolicies
}) {
  if (!getPolicies) {
    return [];
  }
  const errorText = queryString.substring(error.startColumn - 1, error.endColumn - 1);
  const possiblePolicies = await getSpellingPossibilities(getPolicies, errorText);
  return wrapIntoSpellingChangeAction(error, possiblePolicies);
}
async function getSpellingActionForFunctions(error, queryString, ast) {
  const errorText = queryString.substring(error.startColumn - 1, error.endColumn - 1);
  // fallback to the last command if not found
  const commandContext = ast.find(command => error.startColumn > command.location.min && error.startColumn < command.location.max) || ast[ast.length - 1];
  if (!commandContext) {
    return [];
  }
  const possibleSolutions = await getSpellingPossibilities(async () => getCompatibleFunctionDefinitions(commandContext.name, undefined).concat(
  // support nested expressions in STATS
  commandContext.name === 'stats' ? getCompatibleFunctionDefinitions('eval', undefined) : []), errorText.substring(0, errorText.lastIndexOf('(')).toLowerCase() // reduce a bit the distance check making al lowercase
  );
  return wrapIntoSpellingChangeAction(error, possibleSolutions.map(fn => `${fn}${errorText.substring(errorText.lastIndexOf('('))}`));
}
async function getSpellingActionForMetadata(error, queryString, ast, options, {
  getMetaFields
}) {
  if (!getMetaFields) {
    return [];
  }
  const errorText = queryString.substring(error.startColumn - 1, error.endColumn - 1);
  const possibleMetafields = await getSpellingPossibilities(getMetaFields, errorText);
  return wrapIntoSpellingChangeAction(error, possibleMetafields);
}
async function getSpellingActionForEnrichMode(error, queryString, ast, options, _callbacks) {
  var _commandDef$modes;
  const errorText = queryString.substring(error.startColumn - 1, error.endColumn - 1);
  const commandContext = ast.find(command => command.location.max > error.endColumn) || ast[ast.length - 1];
  if (!commandContext) {
    return [];
  }
  const commandDef = (0, _helpers.getCommandDefinition)(commandContext.name);
  const allModes = ((_commandDef$modes = commandDef.modes) === null || _commandDef$modes === void 0 ? void 0 : _commandDef$modes.flatMap(({
    values,
    prefix
  }) => values.map(({
    name
  }) => `${prefix || ''}${name}`))) || [];
  const possibleEnrichModes = await getSpellingPossibilities(async () => allModes, errorText);
  // if no possible solution is found, push all modes
  if (!possibleEnrichModes.length) {
    possibleEnrichModes.push(...allModes);
  }
  return wrapIntoSpellingChangeAction(error, possibleEnrichModes);
}
function wrapIntoSpellingChangeAction(error, possibleSolution) {
  return possibleSolution.map(solution => createAction(
  // @TODO: workout why the tooltip is truncating the title here
  _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.quickfix.replaceWithSolution', {
    defaultMessage: 'Did you mean {solution} ?',
    values: {
      solution
    }
  }), solution, error));
}
function extractQuotedText(rawText, error) {
  return rawText.substring(error.startColumn - 2, error.endColumn);
}
function inferCodeFromError(error, ast, rawText) {
  if (error.message.endsWith('expecting QUOTED_STRING')) {
    const value = extractQuotedText(rawText, error);
    return /^'(.)*'$/.test(value) ? 'wrongQuotes' : undefined;
  }
  if (error.message.startsWith('SyntaxError: token recognition error at:')) {
    // scope it down to column items for now
    const {
      node
    } = (0, _context.getAstContext)(rawText, ast, error.startColumn - 2);
    return node && (0, _helpers.isColumnItem)(node) ? 'quotableFields' : undefined;
  }
}
async function getActions(innerText, markers, astProvider, options = {}, resourceRetriever) {
  const actions = [];
  if (markers.length === 0) {
    return actions;
  }
  const editorMarkers = (0, _utils.wrapAsEditorMessage)('error', markers);
  const {
    ast
  } = await astProvider(innerText);
  const queryForFields = (0, _helpers2.buildQueryForFieldsFromSource)(innerText, ast);
  const {
    getFieldsByType
  } = getFieldsByTypeRetriever(queryForFields, resourceRetriever);
  const getSources = getSourcesRetriever(resourceRetriever);
  const {
    getPolicies,
    getPolicyFields
  } = getPolicyRetriever(resourceRetriever);
  const getMetaFields = getMetaFieldsRetriever(innerText, ast, resourceRetriever);
  const callbacks = {
    getFieldsByType: resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getFieldsFor ? getFieldsByType : undefined,
    getSources: resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getSources ? getSources : undefined,
    getPolicies: resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getPolicies ? getPolicies : undefined,
    getPolicyFields: resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getPolicies ? getPolicyFields : undefined,
    getMetaFields: resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getMetaFields ? getMetaFields : undefined
  };

  // Markers are sent only on hover and are limited to the hovered area
  // so unless there are multiple error/markers for the same area, there's just one
  // in some cases, like syntax + semantic errors (i.e. unquoted fields eval field-1 ), there might be more than one
  for (const error of editorMarkers) {
    var _error$code;
    const code = (_error$code = error.code) !== null && _error$code !== void 0 ? _error$code : inferCodeFromError(error, ast, innerText);
    switch (code) {
      case 'unknownColumn':
        {
          const [columnsSpellChanges, columnsQuotedChanges] = await Promise.all([getSpellingActionForColumns(error, innerText, ast, options, callbacks), getQuotableActionForColumns(error, innerText, ast, options, callbacks)]);
          actions.push(...(columnsQuotedChanges.length ? columnsQuotedChanges : columnsSpellChanges));
          break;
        }
      case 'quotableFields':
        {
          const columnsQuotedChanges = await getQuotableActionForColumns(error, innerText, ast, options, callbacks);
          actions.push(...columnsQuotedChanges);
          break;
        }
      case 'unknownIndex':
        const indexSpellChanges = await getSpellingActionForIndex(error, innerText, ast, options, callbacks);
        actions.push(...indexSpellChanges);
        break;
      case 'unknownPolicy':
        const policySpellChanges = await getSpellingActionForPolicies(error, innerText, ast, options, callbacks);
        actions.push(...policySpellChanges);
        break;
      case 'unknownFunction':
        const fnsSpellChanges = await getSpellingActionForFunctions(error, innerText, ast);
        actions.push(...fnsSpellChanges);
        break;
      case 'unknownMetadataField':
        const metadataSpellChanges = await getSpellingActionForMetadata(error, innerText, ast, options, callbacks);
        actions.push(...metadataSpellChanges);
        break;
      case 'wrongQuotes':
        // it is a syntax error, so location won't be helpful here
        const errorText = extractQuotedText(innerText, error);
        actions.push(createAction(_i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.quickfix.replaceWithQuote', {
          defaultMessage: 'Change quote to " (double)'
        }), errorText.replaceAll("'", '"'),
        // override the location
        {
          ...error,
          startColumn: error.startColumn - 1,
          endColumn: error.startColumn + errorText.length
        }));
        break;
      case 'unsupportedSettingCommandValue':
        const enrichModeSpellChanges = await getSpellingActionForEnrichMode(error, innerText, ast, options, callbacks);
        actions.push(...enrichModeSpellChanges);
        break;
      default:
        break;
    }
  }
  return (0, _lodash.uniqBy)(actions, ({
    edits
  }) => edits[0].text);
}