"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AstListener = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _esql_parser = _interopRequireDefault(require("./antlr/esql_parser"));
var _ast_helpers = require("./ast_helpers");
var _ast_position_utils = require("./ast_position_utils");
var _ast_walker = require("./ast_walker");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class AstListener {
  constructor() {
    (0, _defineProperty2.default)(this, "ast", []);
  }
  getAst() {
    return {
      ast: this.ast
    };
  }

  /**
   * Exit a parse tree produced by the `showInfo`
   * labeled alternative in `esql_parser.showCommand`.
   * @param ctx the parse tree
   */
  exitShowInfo(ctx) {
    const commandAst = (0, _ast_helpers.createCommand)('show', ctx);
    this.ast.push(commandAst);
    commandAst.text = ctx.getText();
    if ((0, _ast_helpers.textExistsAndIsValid)(ctx.INFO().getText())) {
      commandAst === null || commandAst === void 0 ? void 0 : commandAst.args.push((0, _ast_helpers.createFunction)('info', ctx, (0, _ast_position_utils.getPosition)(ctx.INFO().symbol)));
    }
  }

  /**
   * Exit a parse tree produced by the `showFunctions`
   * labeled alternative in `esql_parser.showCommand`.
   * @param ctx the parse tree
   */
  exitMetaFunctions(ctx) {
    const commandAst = (0, _ast_helpers.createCommand)('meta', ctx);
    this.ast.push(commandAst);
    // update the text
    commandAst.text = ctx.getText();
    if ((0, _ast_helpers.textExistsAndIsValid)(ctx.FUNCTIONS().getText())) {
      commandAst === null || commandAst === void 0 ? void 0 : commandAst.args.push((0, _ast_helpers.createFunction)('functions', ctx, (0, _ast_position_utils.getPosition)(ctx.FUNCTIONS().symbol)));
    }
  }

  /**
   * Enter a parse tree produced by `esql_parser.singleStatement`.
   * @param ctx the parse tree
   */
  enterSingleStatement(ctx) {
    this.ast = [];
  }

  /**
   * Exit a parse tree produced by `esql_parser.whereCommand`.
   * @param ctx the parse tree
   */
  exitWhereCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('where', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.collectBooleanExpression)(ctx.booleanExpression()));
  }

  /**
   * Exit a parse tree produced by `esql_parser.rowCommand`.
   * @param ctx the parse tree
   */
  exitRowCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('row', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.collectAllFieldsStatements)(ctx.fields()));
  }

  /**
   * Exit a parse tree produced by `esql_parser.fromCommand`.
   * @param ctx the parse tree
   */
  exitFromCommand(ctx) {
    var _metadataContext$depr;
    const commandAst = (0, _ast_helpers.createCommand)('from', ctx);
    this.ast.push(commandAst);
    commandAst.args.push(...(0, _ast_walker.collectAllSourceIdentifiers)(ctx));
    const metadataContext = ctx.metadata();
    const metadataContent = (metadataContext === null || metadataContext === void 0 ? void 0 : (_metadataContext$depr = metadataContext.deprecated_metadata()) === null || _metadataContext$depr === void 0 ? void 0 : _metadataContext$depr.metadataOption()) || (metadataContext === null || metadataContext === void 0 ? void 0 : metadataContext.metadataOption());
    if (metadataContent) {
      const option = (0, _ast_helpers.createOption)(metadataContent.METADATA().getText().toLowerCase(), metadataContent);
      commandAst.args.push(option);
      option.args.push(...(0, _ast_walker.collectAllColumnIdentifiers)(metadataContent));
    }
  }

  /**
   * Exit a parse tree produced by `esql_parser.evalCommand`.
   * @param ctx the parse tree
   */
  exitEvalCommand(ctx) {
    const commandAst = (0, _ast_helpers.createCommand)('eval', ctx);
    this.ast.push(commandAst);
    commandAst.args.push(...(0, _ast_walker.collectAllFieldsStatements)(ctx.fields()));
  }

  /**
   * Exit a parse tree produced by `esql_parser.statsCommand`.
   * @param ctx the parse tree
   */
  exitStatsCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('stats', ctx);
    this.ast.push(command);

    // STATS expression is optional
    if (ctx._stats) {
      command.args.push(...(0, _ast_walker.collectAllFieldsStatements)(ctx.fields(0)));
    }
    if (ctx._grouping) {
      command.args.push(...(0, _ast_walker.visitByOption)(ctx, ctx._stats ? ctx.fields(1) : ctx.fields(0)));
    }
  }

  /**
   * Exit a parse tree produced by `esql_parser.limitCommand`.
   * @param ctx the parse tree
   */
  exitLimitCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('limit', ctx);
    this.ast.push(command);
    if (ctx.getToken(_esql_parser.default.INTEGER_LITERAL, 0)) {
      const literal = (0, _ast_helpers.createLiteral)('number', ctx.INTEGER_LITERAL());
      if (literal) {
        command.args.push(literal);
      }
    }
  }

  /**
   * Exit a parse tree produced by `esql_parser.sortCommand`.
   * @param ctx the parse tree
   */
  exitSortCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('sort', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.visitOrderExpression)(ctx.orderExpression_list()));
  }

  /**
   * Exit a parse tree produced by `esql_parser.keepCommand`.
   * @param ctx the parse tree
   */
  exitKeepCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('keep', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.collectAllColumnIdentifiers)(ctx));
  }

  /**
   * Exit a parse tree produced by `esql_parser.dropCommand`.
   * @param ctx the parse tree
   */
  exitDropCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('drop', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.collectAllColumnIdentifiers)(ctx));
  }

  /**
   * Exit a parse tree produced by `esql_parser.renameCommand`.
   * @param ctx the parse tree
   */
  exitRenameCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('rename', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.visitRenameClauses)(ctx.renameClause_list()));
  }

  /**
   * Exit a parse tree produced by `esql_parser.dissectCommand`.
   * @param ctx the parse tree
   */
  exitDissectCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('dissect', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.visitDissect)(ctx));
  }

  /**
   * Exit a parse tree produced by `esql_parser.grokCommand`.
   * @param ctx the parse tree
   */
  exitGrokCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('grok', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.visitGrok)(ctx));
  }

  /**
   * Exit a parse tree produced by `esql_parser.mvExpandCommand`.
   * @param ctx the parse tree
   */
  exitMvExpandCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('mv_expand', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.collectAllColumnIdentifiers)(ctx));
  }

  /**
   * Enter a parse tree produced by `esql_parser.showCommand`.
   * @param ctx the parse tree
   */
  enterShowCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('show', ctx);
    this.ast.push(command);
  }

  /**
   * Enter a parse tree produced by `esql_parser.metaCommand`.
   * @param ctx the parse tree
   */
  enterMetaCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('meta', ctx);
    this.ast.push(command);
  }
  /**
   * Exit a parse tree produced by `esql_parser.enrichCommand`.
   * @param ctx the parse tree
   */
  exitEnrichCommand(ctx) {
    const command = (0, _ast_helpers.createCommand)('enrich', ctx);
    this.ast.push(command);
    command.args.push(...(0, _ast_walker.getPolicyName)(ctx), ...(0, _ast_walker.getMatchField)(ctx), ...(0, _ast_walker.getEnrichClauses)(ctx));
  }
  enterEveryRule(ctx) {
    // method not implemented, added to satisfy interface expectation
  }
  visitErrorNode(node) {
    // method not implemented, added to satisfy interface expectation
  }
  visitTerminal(node) {
    // method not implemented, added to satisfy interface expectation
  }
  exitEveryRule(ctx) {
    // method not implemented, added to satisfy interface expectation
  }
}
exports.AstListener = AstListener;