"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchResultProvider = getSearchResultProvider;
exports.toSearchResult = toSearchResult;
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function toSearchResult({
  iconPath,
  isCloud,
  isNative,
  name,
  score,
  serviceType,
  url
}) {
  const isCrawler = serviceType === _constants.ENTERPRISE_SEARCH_CONNECTOR_CRAWLER_SERVICE_TYPE;
  const connectorTypeParam = !isCrawler ? isCloud && isNative ? 'native' : 'connector_client' : null;
  const newUrl = isCrawler ? `${_constants.ENTERPRISE_SEARCH_CONTENT_PLUGIN.URL}/crawlers/new_crawler` : `${_constants.ENTERPRISE_SEARCH_CONTENT_PLUGIN.URL}/connectors/new_connector?connector_type=${connectorTypeParam}&service_type=${serviceType}`;
  return {
    icon: iconPath || 'logoEnterpriseSearch',
    id: serviceType,
    score,
    title: name,
    type: _i18n.i18n.translate('xpack.enterpriseSearch.searchProvider.type.name', {
      defaultMessage: 'Search'
    }),
    url: {
      path: url !== null && url !== void 0 ? url : newUrl,
      prependBasePath: true
    }
  };
}
function getSearchResultProvider(config, connectorTypes, isCloud, crawlerIconPath) {
  return {
    find: ({
      term,
      types,
      tags
    }, {
      aborted$,
      maxResults
    }) => {
      if (tags || types && !(types.includes('integration') || types.includes('enterprise search'))) {
        return (0, _rxjs.from)([[]]);
      }
      const services = [...(config.hasWebCrawler ? [{
        iconPath: crawlerIconPath,
        keywords: ['crawler', 'web', 'website', 'internet', 'google'],
        name: _i18n.i18n.translate('xpack.enterpriseSearch.searchProvider.webCrawler.name', {
          defaultMessage: 'Elastic Web Crawler'
        }),
        serviceType: _constants.ENTERPRISE_SEARCH_CONNECTOR_CRAWLER_SERVICE_TYPE
      }] : []), ...(config.hasConnectors ? connectorTypes : []), ...(config.canDeployEntSearch ? [{
        keywords: ['app', 'search', 'engines'],
        name: _i18n.i18n.translate('xpack.enterpriseSearch.searchProvider.appSearch.name', {
          defaultMessage: 'App Search'
        }),
        serviceType: 'app_search',
        url: _constants.APP_SEARCH_PLUGIN.URL
      }, {
        keywords: ['esre', 'search'],
        name: _i18n.i18n.translate('xpack.enterpriseSearch.searchProvider.aiSearch.name', {
          defaultMessage: 'Search AI'
        }),
        serviceType: 'ai_search',
        url: _constants.AI_SEARCH_PLUGIN.URL
      }] : [])];
      const result = services.map(service => {
        const {
          iconPath,
          isNative,
          keywords,
          name,
          serviceType
        } = service;
        const url = 'url' in service ? service.url : undefined;
        let score = 0;
        const searchTerm = (term || '').toLowerCase();
        const searchName = name.toLowerCase();
        if (!searchTerm) {
          score = 80;
        } else if (searchName === searchTerm) {
          score = 100;
        } else if (searchName.startsWith(searchTerm)) {
          score = 90;
        } else if (searchName.includes(searchTerm)) {
          score = 75;
        } else if (serviceType === searchTerm) {
          score = 65;
        } else if (keywords.some(keyword => keyword.includes(searchTerm))) {
          score = 50;
        }
        return toSearchResult({
          iconPath,
          isCloud,
          isNative,
          name,
          score,
          serviceType,
          url
        });
      }).filter(({
        score
      }) => score > 0).slice(0, maxResults);
      return (0, _rxjs.from)([result]).pipe((0, _rxjs.takeUntil)(aborted$));
    },
    getSearchableTypes: () => ['enterprise search', 'integration'],
    id: 'enterpriseSearch'
  };
}