"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.preparePipelineAndIndexForMlInference = exports.createMlInferencePipeline = void 0;
var _error_codes = require("../../../../../../common/types/error_codes");
var _create_ml_inference_pipeline = require("../../../../../utils/create_ml_inference_pipeline");
var _ml_inference_pipeline_utils = require("../../../../../utils/ml_inference_pipeline_utils");
var _update_ml_inference_mappings = require("../update_ml_inference_mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates a Machine Learning Inference pipeline with the given settings, if it doesn't exist yet,
 * then references it in the "parent" ML Inference pipeline that is associated with the index.
 * Finally, updates the index's mappings to accommodate the specified outputs of the inference model (if able)
 * @param indexName name of the index this pipeline corresponds to.
 * @param pipelineName pipeline name set by the user.
 * @param pipelineDefinition
 * @param modelId model ID selected by the user.
 * @param fieldMappings The array of objects representing the source field (text) names and target fields (ML output) names
 * @param esClient the Elasticsearch Client to use when retrieving pipeline and model details.
 */
const preparePipelineAndIndexForMlInference = async (indexName, pipelineName, pipelineDefinition, modelId, fieldMappings, esClient) => {
  const createPipelineResult = await createMlInferencePipeline(pipelineName, pipelineDefinition, esClient);
  const addSubPipelineResult = await (0, _create_ml_inference_pipeline.addSubPipelineToIndexSpecificMlPipeline)(indexName, createPipelineResult.id, esClient);
  const mappingResponse = fieldMappings ? (await (0, _update_ml_inference_mappings.updateMlInferenceMappings)(indexName, modelId, fieldMappings, esClient)).acknowledged : false;
  return {
    added_to_parent_pipeline: addSubPipelineResult.addedToParentPipeline,
    created_pipeline: createPipelineResult.created,
    mapping_updated: mappingResponse,
    pipeline_id: createPipelineResult.id
  };
};

/**
 * Creates a Machine Learning Inference pipeline with the given settings, if it doesn't exist yet.
 * @param pipelineName pipeline name set by the user.
 * @param pipelineDefinition full definition of the pipeline
 * @param esClient the Elasticsearch Client to use when retrieving pipeline and model details.
 */
exports.preparePipelineAndIndexForMlInference = preparePipelineAndIndexForMlInference;
const createMlInferencePipeline = async (pipelineName, pipelineDefinition, esClient) => {
  var _pipelineByName;
  const inferencePipelineGeneratedName = (0, _ml_inference_pipeline_utils.getPrefixedInferencePipelineProcessorName)(pipelineName);

  // Check that a pipeline with the same name doesn't already exist
  let pipelineByName;
  try {
    pipelineByName = await esClient.ingest.getPipeline({
      id: inferencePipelineGeneratedName
    });
  } catch (error) {
    // Silently swallow error
  }
  if ((_pipelineByName = pipelineByName) !== null && _pipelineByName !== void 0 && _pipelineByName[inferencePipelineGeneratedName]) {
    throw new Error(_error_codes.ErrorCode.PIPELINE_ALREADY_EXISTS);
  }

  // TODO: See if we can defer this error handling to putPipeline()
  if (!pipelineDefinition) {
    throw new Error(_error_codes.ErrorCode.PARAMETER_CONFLICT);
  }
  await esClient.ingest.putPipeline({
    id: inferencePipelineGeneratedName,
    ...pipelineDefinition,
    version: 1
  });
  return {
    created: true,
    id: inferencePipelineGeneratedName
  };
};
exports.createMlInferencePipeline = createMlInferencePipeline;