"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findUserConversationsRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _find_conversations_route = require("@kbn/elastic-assistant-common/impl/schemas/conversations/find_conversations_route.gen");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _utils = require("../utils");
var _transforms = require("../../ai_assistant_data_clients/conversations/transforms");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findUserConversationsRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_CONVERSATIONS_URL_FIND,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        query: (0, _common.buildRouteValidationWithZod)(_find_conversations_route.FindConversationsRequestQuery)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    try {
      const {
        query
      } = request;
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
      const license = ctx.licensing.license;
      if (!(0, _helpers.hasAIAssistantLicense)(license)) {
        return response.forbidden({
          body: {
            message: _helpers.UPGRADE_LICENSE_MESSAGE
          }
        });
      }
      const dataClient = await ctx.elasticAssistant.getAIAssistantConversationsDataClient();
      const currentUser = ctx.elasticAssistant.getCurrentUser();
      const additionalFilter = query.filter ? ` AND ${query.filter}` : '';
      const result = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.findDocuments({
        perPage: query.per_page,
        page: query.page,
        sortField: query.sort_field,
        sortOrder: query.sort_order,
        filter: `users:{ id: "${currentUser === null || currentUser === void 0 ? void 0 : currentUser.profile_uid}" }${additionalFilter}`,
        fields: query.fields
      }));
      if (result) {
        return response.ok({
          body: {
            perPage: result.perPage,
            page: result.page,
            total: result.total,
            data: (0, _transforms.transformESSearchToConversations)(result.data)
          }
        });
      }
      return response.ok({
        body: {
          perPage: query.per_page,
          page: query.page,
          data: [],
          total: 0
        }
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.findUserConversationsRoute = findUserConversationsRoute;