"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteKnowledgeBaseRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _crud_kb_route = require("@kbn/elastic-assistant-common/impl/schemas/knowledge_base/crud_kb_route.gen");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _build_response = require("../../lib/build_response");
var _constants = require("../../../common/constants");
var _elasticsearch_store = require("../../lib/langchain/elasticsearch_store/elasticsearch_store");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Delete Knowledge Base index, pipeline, and resources (collection of documents)
 * @param router
 */
const deleteKnowledgeBaseRoute = router => {
  router.versioned.delete({
    access: 'internal',
    path: _constants.KNOWLEDGE_BASE,
    options: {
      // Note: Relying on current user privileges to scope an esClient.
      // Add `access:kbnElasticAssistant` to limit API access to only users with assistant privileges
      tags: []
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_crud_kb_route.DeleteKnowledgeBaseRequestParams)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    try {
      const kbResource = request.params.resource != null ? decodeURIComponent(request.params.resource) : undefined;

      // Get a scoped esClient for deleting the Knowledge Base index, pipeline, and documents
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const esStore = new _elasticsearch_store.ElasticsearchStore(esClient, _constants2.KNOWLEDGE_BASE_INDEX_PATTERN, logger, telemetry);
      if (kbResource === _constants2.ESQL_RESOURCE) {
        // For now, tearing down the Knowledge Base is fine, but will want to support removing specific assets based
        // on resource name or document query
        // Implement deleteDocuments(query: string) in ElasticsearchStore
        // const success = await esStore.deleteDocuments();
        // return const body: DeleteKnowledgeBaseResponse = { success };
      }

      // Delete index and pipeline
      const indexDeleted = await esStore.deleteIndex();
      const pipelineDeleted = await esStore.deletePipeline();
      const body = {
        success: indexDeleted && pipelineDeleted
      };
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.deleteKnowledgeBaseRoute = deleteKnowledgeBaseRoute;