"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postAttackDiscoveryRoute = void 0;
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _language_models = require("@kbn/elastic-assistant-common/impl/language_models");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../../common/constants");
var _helpers = require("./helpers");
var _helpers2 = require("../helpers");
var _utils = require("../evaluate/utils");
var _build_response = require("../../lib/build_response");
var _utils2 = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROUTE_HANDLER_TIMEOUT = 10 * 60 * 1000; // 10 * 60 seconds = 10 minutes
const LANG_CHAIN_TIMEOUT = ROUTE_HANDLER_TIMEOUT - 10_000; // 9 minutes 50 seconds
const CONNECTOR_TIMEOUT = LANG_CHAIN_TIMEOUT - 10_000; // 9 minutes 40 seconds

const postAttackDiscoveryRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _constants.ATTACK_DISCOVERY,
    options: {
      tags: ['access:elasticAssistant'],
      timeout: {
        idleSocket: ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryPostRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryPostResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    try {
      // get the actions plugin start contract from the request context:
      const actions = (await context.elasticAssistant).actions;
      const pluginName = (0, _helpers2.getPluginNameFromRequest)({
        request,
        defaultPluginName: _helpers2.DEFAULT_PLUGIN_NAME,
        logger
      });

      // get parameters from the request body
      const alertsIndexPattern = decodeURIComponent(request.body.alertsIndexPattern);
      const connectorId = decodeURIComponent(request.body.connectorId);
      const {
        actionTypeId,
        anonymizationFields,
        langSmithApiKey,
        langSmithProject,
        replacements,
        size
      } = request.body;

      // get an Elasticsearch client for the authenticated user:
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;

      // callback to accumulate the latest replacements:
      let latestReplacements = {
        ...replacements
      };
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };

      // get the attack discovery tool:
      const assistantTools = (await context.elasticAssistant).getRegisteredTools(pluginName);
      const assistantTool = assistantTools.find(tool => tool.id === 'attack-discovery');
      if (!assistantTool) {
        return response.notFound(); // attack discovery tool not found
      }
      const traceOptions = {
        projectName: langSmithProject,
        tracers: [...(0, _utils.getLangSmithTracer)({
          apiKey: langSmithApiKey,
          projectName: langSmithProject,
          logger
        })]
      };
      const llm = new _language_models.ActionsClientLlm({
        actions,
        connectorId,
        llmType: (0, _utils2.getLlmType)(actionTypeId),
        logger,
        request,
        temperature: 0,
        // zero temperature for attack discovery, because we want structured JSON output
        timeout: CONNECTOR_TIMEOUT,
        traceOptions
      });
      const assistantToolParams = (0, _helpers.getAssistantToolParams)({
        alertsIndexPattern,
        anonymizationFields,
        esClient,
        latestReplacements,
        langChainTimeout: LANG_CHAIN_TIMEOUT,
        llm,
        onNewReplacements,
        request,
        size
      });

      // invoke the attack discovery tool:
      const toolInstance = assistantTool.getTool(assistantToolParams);
      const rawAttackDiscoveries = await (toolInstance === null || toolInstance === void 0 ? void 0 : toolInstance.invoke(''));
      if (rawAttackDiscoveries == null) {
        return response.customError({
          body: {
            message: 'tool returned no attack discoveries'
          },
          statusCode: 500
        });
      }
      const {
        alertsContextCount,
        attackDiscoveries
      } = JSON.parse(rawAttackDiscoveries);
      return response.ok({
        body: {
          alertsContextCount,
          attackDiscoveries,
          connector_id: connectorId,
          replacements: latestReplacements
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postAttackDiscoveryRoute = postAttackDiscoveryRoute;