"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleStreamStorage = void 0;
var _promises = require("stream/promises");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const handleStreamStorage = async ({
  abortSignal,
  responseStream,
  actionTypeId,
  onMessageSent,
  logger
}) => {
  try {
    const parser = actionTypeId === '.bedrock' ? parseBedrockStream : parseOpenAIStream;
    const parsedResponse = await parser(responseStream, logger, abortSignal);
    if (onMessageSent) {
      onMessageSent(parsedResponse);
    }
  } catch (e) {
    if (onMessageSent) {
      onMessageSent(`An error occurred while streaming the response:\n\n${e.message}`);
    }
  }
};
exports.handleStreamStorage = handleStreamStorage;
const parseOpenAIStream = async (stream, logger, abortSignal) => {
  let responseBody = '';
  stream.on('data', chunk => {
    responseBody += chunk.toString();
  });
  return new Promise((resolve, reject) => {
    stream.on('end', () => {
      resolve(parseOpenAIResponse(responseBody));
    });
    stream.on('error', err => {
      reject(err);
    });
    if (abortSignal) {
      abortSignal.addEventListener('abort', () => {
        stream.destroy();
        resolve(parseOpenAIResponse(responseBody));
      });
    }
  });
};
const parseOpenAIResponse = responseBody => responseBody.split('\n').filter(line => {
  return line.startsWith('data: ') && !line.endsWith('[DONE]');
}).map(line => {
  return JSON.parse(line.replace('data: ', ''));
}).filter(line => {
  return 'object' in line && line.object === 'chat.completion.chunk';
}).reduce((prev, line) => {
  const msg = line.choices[0].delta;
  return prev + (msg.content || '');
}, '');
const parseBedrockStream = async (responseStream, logger, abortSignal) => {
  const responseBuffer = [];
  if (abortSignal) {
    abortSignal.addEventListener('abort', () => {
      responseStream.destroy(new Error('Aborted'));
      return parseBedrockBuffer(responseBuffer, logger);
    });
  }
  responseStream.on('data', chunk => {
    // special encoding for bedrock, do not attempt to convert to string
    responseBuffer.push(chunk);
  });
  await (0, _promises.finished)(responseStream).catch(err => {
    if (abortSignal !== null && abortSignal !== void 0 && abortSignal.aborted) {
      logger.info('Bedrock stream parsing was aborted.');
    } else {
      throw err;
    }
  });
  return parseBedrockBuffer(responseBuffer, logger);
};

/**
 * Parses a Bedrock buffer from an array of chunks.
 *
 * @param {Uint8Array[]} chunks - Array of Uint8Array chunks to be parsed.
 * @returns {string} - Parsed string from the Bedrock buffer.
 */
const parseBedrockBuffer = (chunks, logger) => {
  // Initialize an empty Uint8Array to store the concatenated buffer.
  let bedrockBuffer = new Uint8Array(0);

  // Map through each chunk to process the Bedrock buffer.
  return chunks.map(chunk => {
    const processedChunk = (0, _elasticAssistantCommon.handleBedrockChunk)({
      chunk,
      bedrockBuffer,
      logger
    });
    bedrockBuffer = processedChunk.bedrockBuffer;
    return processedChunk.decodedChunk;
  }).join('');
};