"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadESQL = void 0;
var _directory = require("langchain/document_loaders/fs/directory");
var _text = require("langchain/document_loaders/fs/text");
var _path = require("path");
var _add_required_kb_resource_metadata = require("./add_required_kb_resource_metadata");
var _constants = require("../../../routes/knowledge_base/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Loads the ESQL docs and language files into the Knowledge Base.
 *
 * *Item of Interest*
 * Knob #1: Types of documents loaded, metadata included, and document chunking strategies + text-splitting
 */
const loadESQL = async (esStore, logger) => {
  try {
    var _response$length;
    const docsLoader = new _directory.DirectoryLoader((0, _path.resolve)(__dirname, '../../../knowledge_base/esql/documentation'), {
      '.asciidoc': path => new _text.TextLoader(path)
    }, true);
    const languageLoader = new _directory.DirectoryLoader((0, _path.resolve)(__dirname, '../../../knowledge_base/esql/language_definition'), {
      '.g4': path => new _text.TextLoader(path),
      '.tokens': path => new _text.TextLoader(path)
    }, true);
    const exampleQueriesLoader = new _directory.DirectoryLoader((0, _path.resolve)(__dirname, '../../../knowledge_base/esql/example_queries'), {
      '.asciidoc': path => new _text.TextLoader(path)
    }, true);
    const docs = await docsLoader.load();
    const languageDocs = await languageLoader.load();
    const rawExampleQueries = await exampleQueriesLoader.load();

    // Add additional metadata to the example queries that indicates they are required KB documents:
    const requiredExampleQueries = (0, _add_required_kb_resource_metadata.addRequiredKbResourceMetadata)({
      docs: rawExampleQueries,
      kbResource: _constants.ESQL_RESOURCE
    });
    logger.info(`Loading ${docs.length} ES|QL docs, ${languageDocs.length} language docs, and ${requiredExampleQueries.length} example queries into the Knowledge Base`);
    const response = await esStore.addDocuments([...docs, ...languageDocs, ...requiredExampleQueries]);
    logger.info(`Loaded ${(_response$length = response === null || response === void 0 ? void 0 : response.length) !== null && _response$length !== void 0 ? _response$length : 0} ES|QL docs, language docs, and example queries into the Knowledge Base`);
    return response.length > 0;
  } catch (e) {
    logger.error(`Failed to load ES|QL docs, language docs, and example queries into the Knowledge Base\n${e}`);
    return false;
  }
};
exports.loadESQL = loadESQL;