"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postResultsRoute = void 0;
var _constants = require("../../../common/constants");
var _build_response = require("../../lib/build_response");
var _common = require("../../schemas/common");
var _result = require("../../schemas/result");
var _translations = require("../../translations");
var _privileges = require("./privileges");
var _translations2 = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postResultsRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.RESULTS_ROUTE_PATH,
    access: 'internal',
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        body: (0, _common.buildRouteValidation)(_result.PostResultBody)
      }
    }
  }, async (context, request, response) => {
    const services = await context.resolve(['core', 'dataQualityDashboard']);
    const resp = (0, _build_response.buildResponse)(response);
    let index;
    try {
      index = await services.dataQualityDashboard.getResultsIndexName();
    } catch (err) {
      logger.error(`[POST result] Error retrieving results index name: ${err.message}`);
      return resp.error({
        body: `${_translations2.API_RESULTS_INDEX_NOT_AVAILABLE}: ${err.message}`,
        statusCode: 503
      });
    }
    const currentUser = services.core.security.authc.getCurrentUser();
    if (!currentUser) {
      return resp.error({
        body: _translations.API_CURRENT_USER_ERROR_MESSAGE,
        statusCode: 500
      });
    }
    try {
      var _indicesResponse$inde;
      const {
        client
      } = services.core.elasticsearch;
      const {
        indexName
      } = request.body;

      // Confirm index exists and get the data stream name if it's a data stream
      const indicesResponse = await client.asInternalUser.indices.get({
        index: indexName,
        features: 'aliases'
      });
      if (!indicesResponse[indexName]) {
        return response.ok({
          body: {
            result: 'noop'
          }
        });
      }
      const indexOrDataStream = (_indicesResponse$inde = indicesResponse[indexName].data_stream) !== null && _indicesResponse$inde !== void 0 ? _indicesResponse$inde : indexName;

      // Confirm user has authorization for the index name or data stream
      const hasIndexPrivileges = await (0, _privileges.checkIndicesPrivileges)({
        client,
        indices: [indexOrDataStream]
      });
      if (!hasIndexPrivileges[indexOrDataStream]) {
        return response.ok({
          body: {
            result: 'noop'
          }
        });
      }

      // Index the result
      const body = {
        ...request.body,
        '@timestamp': Date.now(),
        checkedBy: currentUser.profile_uid
      };
      const outcome = await client.asInternalUser.index({
        index,
        body
      });
      return response.ok({
        body: {
          result: outcome.result
        }
      });
    } catch (err) {
      var _err$message, _err$statusCode;
      logger.error(err.message);
      return resp.error({
        body: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : _translations.API_DEFAULT_ERROR_MESSAGE,
        statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500
      });
    }
  });
};
exports.postResultsRoute = postResultsRoute;