"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateSavedSearch = updateSavedSearch;
var _lodash = require("lodash");
var _is_text_based_query = require("./is_text_based_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Updates the saved search with a given data view & Appstate
 * Is executed on every change of those, for making sure the saved search is
 * up to date before fetching data and persisting or sharing
 * @param savedSearch
 * @param dataView
 * @param state
 * @param globalStateContainer
 * @param services
 * @param useFilterAndQueryServices - when true data services are being used for updating filter + query
 */
function updateSavedSearch({
  savedSearch,
  dataView,
  state,
  globalStateContainer,
  services,
  useFilterAndQueryServices = false
}) {
  var _savedSearch$searchSo;
  if (dataView && ((_savedSearch$searchSo = savedSearch.searchSource.getField('index')) === null || _savedSearch$searchSo === void 0 ? void 0 : _savedSearch$searchSo.id) !== dataView.id) {
    savedSearch.searchSource.setField('index', dataView);
    if (!dataView.isPersisted()) {
      savedSearch.usesAdHocDataView = true;
    }
  }
  if (useFilterAndQueryServices) {
    savedSearch.searchSource.setField('query', services.data.query.queryString.getQuery()).setField('filter', services.data.query.filterManager.getFilters());
  } else if (state) {
    var _globalStateContainer, _globalStateContainer2, _state$query;
    const appFilters = state.filters ? (0, _lodash.cloneDeep)(state.filters) : [];
    const globalFilters = (_globalStateContainer = (_globalStateContainer2 = globalStateContainer.get()) === null || _globalStateContainer2 === void 0 ? void 0 : _globalStateContainer2.filters) !== null && _globalStateContainer !== void 0 ? _globalStateContainer : [];
    savedSearch.searchSource.setField('query', (_state$query = state.query) !== null && _state$query !== void 0 ? _state$query : undefined).setField('filter', [...globalFilters, ...appFilters]);
  }
  if (state) {
    savedSearch.columns = state.columns || [];
    savedSearch.sort = state.sort || [];
    if (state.grid) {
      savedSearch.grid = state.grid;
    }
    savedSearch.hideChart = state.hideChart;
    savedSearch.rowHeight = state.rowHeight;
    savedSearch.headerRowHeight = state.headerRowHeight;
    savedSearch.rowsPerPage = state.rowsPerPage;
    savedSearch.sampleSize = state.sampleSize;
    if (state.viewMode) {
      savedSearch.viewMode = state.viewMode;
    }
    if (typeof state.breakdownField !== 'undefined') {
      savedSearch.breakdownField = state.breakdownField;
    } else if (savedSearch.breakdownField) {
      savedSearch.breakdownField = '';
    }
    savedSearch.hideAggregatedPreview = state.hideAggregatedPreview;

    // add a flag here to identify text based language queries
    // these should be filtered out from the visualize editor
    const isTextBasedQueryResult = (0, _is_text_based_query.isTextBasedQuery)(state.query);
    if (savedSearch.isTextBasedQuery || isTextBasedQueryResult) {
      savedSearch.isTextBasedQuery = isTextBasedQueryResult;
    }
  }
  const {
    from,
    to
  } = services.timefilter.getTime();
  const refreshInterval = services.timefilter.getRefreshInterval();
  savedSearch.timeRange = savedSearch.timeRestore || savedSearch.timeRange ? {
    from,
    to
  } : undefined;
  savedSearch.refreshInterval = savedSearch.timeRestore || savedSearch.refreshInterval ? {
    value: refreshInterval.value,
    pause: refreshInterval.pause
  } : undefined;
  return savedSearch;
}