"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSort = getSort;
exports.getSortArray = getSortArray;
exports.isLegacySort = isLegacySort;
exports.isSortable = isSortable;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function isSortable(fieldName, dataView, isTextBasedQueryMode) {
  if (isTextBasedQueryMode) {
    // in-memory sorting is used for text-based queries
    // would be great to have a way to determine if a text-based column is sortable
    return fieldName !== '_source';
  }
  const field = dataView.getFieldByName(fieldName);
  return !!(field && field.sortable);
}
function createSortObject(sortPair, dataView, isTextBasedQueryMode) {
  if (Array.isArray(sortPair) && sortPair.length === 2 && isSortable(String(sortPair[0]), dataView, isTextBasedQueryMode)) {
    const [field, direction] = sortPair;
    return {
      [field]: direction
    };
  } else if ((0, _lodash.isPlainObject)(sortPair) && isSortable(Object.keys(sortPair)[0], dataView, isTextBasedQueryMode)) {
    return sortPair;
  }
}
function isLegacySort(sort) {
  return sort.length === 2 && typeof sort[0] === 'string' && (sort[1] === 'desc' || sort[1] === 'asc');
}

/**
 * Take a sorting array and make it into an object
 * @param {array} sort two dimensional array [[fieldToSort, directionToSort]]
 *  or an array of objects [{fieldToSort: directionToSort}]
 * @param {object} dataView used for determining default sort
 * @param {boolean} isTextBasedQueryMode
 * @returns Array<{object}> an array of sort objects
 */
function getSort(sort, dataView, isTextBasedQueryMode) {
  if (Array.isArray(sort)) {
    if (isLegacySort(sort)) {
      // To stay compatible with legacy sort, which just supported a single sort field
      return [{
        [sort[0]]: sort[1]
      }];
    }
    return sort.map(sortPair => createSortObject(sortPair, dataView, isTextBasedQueryMode)).filter(sortPairObj => typeof sortPairObj === 'object');
  }
  return [];
}

/**
 * compared to getSort it doesn't return an array of objects, it returns an array of arrays
 * [[fieldToSort: directionToSort]]
 */
function getSortArray(sort, dataView, isTextBasedQueryMode) {
  return getSort(sort, dataView, isTextBasedQueryMode).reduce((acc, sortPair) => {
    const entries = Object.entries(sortPair);
    if (entries && entries[0]) {
      acc.push(entries[0]);
    }
    return acc;
  }, []);
}