"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQLNumericFieldStats = void 0;
var _common = require("@kbn/data-plugin/common");
var _esqlUtils = require("@kbn/esql-utils");
var _lodash = require("lodash");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _process_distribution_data = require("../../utils/process_distribution_data");
var _constants = require("../requests/constants");
var _esql_utils = require("../requests/esql_utils");
var _promise_all_settled_utils = require("../../../common/util/promise_all_settled_utils");
var _index_data_visualizer_viewer = require("../../constants/index_data_visualizer_viewer");
var _handle_error = require("./handle_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getESQLNumericFieldStatsInChunk = async ({
  runRequest,
  columns,
  esqlBaseQuery,
  filter
}) => {
  // Hashmap of agg to index/order of which is made in the ES|QL query
  // {min: 0, max: 1, p0: 2, p5: 3, ..., p100: 22}
  const numericAccessorMap = _esql_utils.PERCENTS.reduce((acc, curr, idx) => {
    // +2 for the min and max aggs
    acc[`p${curr}`] = idx + 2;
    return acc;
  }, {
    // First two are min and max aggs
    min: 0,
    max: 1
    // and percentiles p0, p5, ..., p100 are the rest
  });
  const numericFields = columns.map((field, idx) => {
    const percentiles = (0, _esql_utils.getESQLPercentileQueryArray)(field.name, _esql_utils.PERCENTS);
    return {
      field,
      query: `${(0, _esql_utils.getSafeESQLName)(`${field.name}_min`)} = MIN(${(0, _esql_utils.getSafeESQLName)(field.name)}),
    ${(0, _esql_utils.getSafeESQLName)(`${field.name}_max`)} = MAX(${(0, _esql_utils.getSafeESQLName)(field.name)}),
  ${percentiles.join(',')}
  `,
      // Start index of field in the response, so we know to slice & access the values
      startIndex: idx * Object.keys(numericAccessorMap).length
    };
  });
  if (numericFields.length > 0) {
    const numericStatsQuery = '| STATS ' + numericFields.map(({
      query
    }) => query).join(',');
    const query = (0, _esqlUtils.appendToESQLQuery)(esqlBaseQuery, numericStatsQuery);
    const request = {
      params: {
        query,
        ...(filter ? {
          filter
        } : {})
      }
    };
    try {
      const fieldStatsResp = await runRequest(request, {
        strategy: _common.ESQL_SEARCH_STRATEGY
      });
      if (fieldStatsResp) {
        const values = fieldStatsResp.rawResponse.values[0];
        return numericFields.map(({
          field,
          startIndex
        }, idx) => {
          /** Order of aggs we are expecting back from query
           * 0 = min; 23 = startIndex + 0 for 2nd field
           * 1 = max; 24 = startIndex + 1
           * 2 p0; 25; 24 = startIndex + 2
           * 3 p5; 26
           * 4 p10; 27
           * ...
           * 22 p100;
           */
          const min = values[startIndex + numericAccessorMap.min];
          const max = values[startIndex + numericAccessorMap.max];
          const median = values[startIndex + numericAccessorMap.p50];
          const percentiles = values.slice(startIndex + numericAccessorMap.p5, startIndex + numericAccessorMap.p100 + 1).map(value => ({
            value
          }));
          const distribution = (0, _process_distribution_data.processDistributionData)(percentiles, _constants.PERCENTILE_SPACING, min);
          return {
            fieldName: field.name,
            ...field,
            min,
            max,
            median,
            distribution
          };
        });
      }
    } catch (error) {
      (0, _handle_error.handleError)({
        error,
        request
      });
      return numericFields.map(({
        field
      }) => {
        return {
          fieldName: field.name,
          error
        };
      });
    }
  }
  return [];
};
const getESQLNumericFieldStats = async ({
  runRequest,
  columns,
  esqlBaseQuery,
  filter
}) => {
  const limiter = (0, _pLimit.default)(_index_data_visualizer_viewer.MAX_CONCURRENT_REQUESTS);

  // Breakdown so that each requests only contains 10 numeric fields
  // to prevent potential circuit breaking exception
  // or too big of a payload
  const numericColumnChunks = (0, _lodash.chunk)(columns, 10);
  const numericStats = await Promise.allSettled(numericColumnChunks.map(numericColumns => limiter(() => getESQLNumericFieldStatsInChunk({
    columns: numericColumns,
    filter,
    runRequest,
    esqlBaseQuery
  }))));
  return numericStats.filter(_promise_all_settled_utils.isFulfilled).flatMap(stat => stat.value);
};
exports.getESQLNumericFieldStats = getESQLNumericFieldStats;