"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFieldStatsSearchStrategy = useFieldStatsSearchStrategy;
var _react = require("react");
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _eui = require("@elastic/eui");
var _mlQueryUtils = require("@kbn/ml-query-utils");
var _kibana_context = require("../../kibana_context");
var _progress_utils = require("../progress_utils");
var _constants = require("../search_strategy/requests/constants");
var _get_fields_stats = require("../search_strategy/requests/get_fields_stats");
var _index_data_visualizer_viewer = require("../constants/index_data_visualizer_viewer");
var _filter_fields = require("../../common/components/fields_stats_grid/filter_fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createBatchedRequests = (fields, maxBatchSize = 10) => {
  // Batch up fields by type, getting stats for multiple fields at a time.
  const batches = [];
  const batchedFields = {};
  fields.forEach(field => {
    const fieldType = field.type;
    if (batchedFields[fieldType] === undefined) {
      batchedFields[fieldType] = [[]];
    }
    let lastArray = (0, _lodash.last)(batchedFields[fieldType]);
    if (lastArray.length === maxBatchSize) {
      lastArray = [];
      batchedFields[fieldType].push(lastArray);
    }
    lastArray.push(field);
  });
  Object.values(batchedFields).forEach(lists => {
    batches.push(...lists);
  });
  return batches;
};
function useFieldStatsSearchStrategy(searchStrategyParams, fieldStatsParams, dataVisualizerListState, samplingProbability) {
  const {
    services: {
      data,
      notifications: {
        toasts
      }
    }
  } = (0, _kibana_context.useDataVisualizerKibana)();
  const [fieldStats, setFieldStats] = (0, _react.useState)();
  const [fetchState, setFetchState] = (0, _react.useReducer)((0, _progress_utils.getReducer)(), (0, _progress_utils.getInitialProgress)());
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const searchSubscription$ = (0, _react.useRef)();
  const retries$ = (0, _react.useRef)();
  const startFetch = (0, _react.useCallback)(() => {
    var _searchSubscription$$, _retries$$current, _filteredItems$filter;
    (_searchSubscription$$ = searchSubscription$.current) === null || _searchSubscription$$ === void 0 ? void 0 : _searchSubscription$$.unsubscribe();
    (_retries$$current = retries$.current) === null || _retries$$current === void 0 ? void 0 : _retries$$current.unsubscribe();
    abortCtrl.current.abort();
    abortCtrl.current = new AbortController();
    setFetchState({
      ...(0, _progress_utils.getInitialProgress)(),
      error: undefined
    });
    setFieldStats(undefined);
    if (!searchStrategyParams || !fieldStatsParams || fieldStatsParams.metricConfigs.length === 0 && fieldStatsParams.nonMetricConfigs.length === 0) {
      setFetchState({
        loaded: 100,
        isRunning: false
      });
      return;
    }
    const {
      sortField,
      sortDirection
    } = dataVisualizerListState;
    /**
     * Sort the list of fields by the initial sort field and sort direction
     * Then divide into chunks by the initial page size
     */

    const itemsSorter = _eui.Comparators.property(sortField, _eui.Comparators.default(sortDirection));
    const preslicedSortedConfigs = [...fieldStatsParams.metricConfigs, ...fieldStatsParams.nonMetricConfigs].sort(itemsSorter);
    const filteredItems = (0, _filter_fields.filterFields)(preslicedSortedConfigs, dataVisualizerListState.visibleFieldNames, dataVisualizerListState.visibleFieldTypes);
    const {
      pageIndex,
      pageSize
    } = dataVisualizerListState;
    const pageOfConfigs = (_filteredItems$filter = filteredItems.filteredFields) === null || _filteredItems$filter === void 0 ? void 0 : _filteredItems$filter.slice(pageIndex * pageSize, (pageIndex + 1) * pageSize).filter(d => d.existsInDocs === true);
    if (!pageOfConfigs || pageOfConfigs.length === 0) {
      setFetchState({
        loaded: 100,
        isRunning: false
      });
      return;
    }
    const filterCriteria = (0, _mlQueryUtils.buildBaseFilterCriteria)(searchStrategyParams.timeFieldName, searchStrategyParams.earliest, searchStrategyParams.latest, searchStrategyParams.searchQuery);
    const params = {
      index: searchStrategyParams.index,
      timeFieldName: searchStrategyParams.timeFieldName,
      earliestMs: searchStrategyParams.earliest,
      latestMs: searchStrategyParams.latest,
      runtimeFieldMap: searchStrategyParams.runtimeFieldMap,
      intervalMs: searchStrategyParams.intervalMs,
      query: {
        bool: {
          filter: filterCriteria
        }
      },
      maxExamples: _constants.MAX_EXAMPLES_DEFAULT,
      samplingProbability,
      browserSessionSeed: searchStrategyParams.browserSessionSeed,
      samplingOption: searchStrategyParams.samplingOption
    };
    const {
      sessionId,
      embeddableExecutionContext
    } = searchStrategyParams;
    const searchOptions = {
      abortSignal: abortCtrl.current.signal,
      sessionId,
      ...(embeddableExecutionContext ? {
        executionContext: embeddableExecutionContext
      } : {})
    };
    const batches = createBatchedRequests(pageOfConfigs.map((config, idx) => ({
      ...config,
      safeFieldName: (0, _mlQueryUtils.getSafeAggregationName)(config.fieldName, idx)
    })), 10);
    const statsMap$ = new _rxjs.Subject();
    const fieldsToRetry$ = new _rxjs.Subject();
    const fieldStatsToFetch = batches.map(batch => (0, _get_fields_stats.getFieldsStats)(data.search, params, batch, searchOptions)).filter(obs => obs !== undefined);
    const onError = error => {
      toasts.addError(error, {
        title: _i18n.i18n.translate('xpack.dataVisualizer.index.errorFetchingFieldStatisticsMessage', {
          defaultMessage: 'Error fetching field statistics'
        })
      });
      setFetchState({
        isRunning: false,
        error
      });
    };
    const onComplete = () => {
      setFetchState({
        isRunning: false
      });
    };
    const statsMapTmp = new Map();

    // First, attempt to fetch field stats in batches of 10
    searchSubscription$.current = (0, _rxjs.from)(fieldStatsToFetch).pipe((0, _rxjs.mergeMap)(observable => observable, _index_data_visualizer_viewer.MAX_CONCURRENT_REQUESTS)).subscribe({
      next: batchResponse => {
        setFetchState({
          ...(0, _progress_utils.getInitialProgress)(),
          error: undefined
        });
        if (batchResponse) {
          const failedFields = [];
          if (Array.isArray(batchResponse)) {
            batchResponse.forEach(f => {
              if (f.fieldName !== undefined) {
                statsMapTmp.set(f.fieldName, f);
              }
            });
          } else {
            var _batchResponse$fields;
            // If an error occurred during batch
            // retry each field in the failed batch individually
            failedFields.push(...((_batchResponse$fields = batchResponse.fields) !== null && _batchResponse$fields !== void 0 ? _batchResponse$fields : []));
          }
          setFieldStats(statsMapTmp);
          setFetchState({
            loaded: statsMapTmp.size / pageOfConfigs.length * 100,
            isRunning: true
          });
          if (failedFields.length > 0) {
            statsMap$.next(statsMapTmp);
            fieldsToRetry$.next(failedFields);
          }
        }
      },
      error: onError,
      complete: onComplete
    });

    // If any of batches failed, retry each of the failed field at least one time individually
    retries$.current = (0, _rxjs.combineLatest)([statsMap$, fieldsToRetry$.pipe((0, _rxjs.switchMap)(failedFields => {
      return (0, _rxjs.combineLatest)(failedFields.map(failedField => (0, _get_fields_stats.getFieldsStats)(data.search, params, [failedField], searchOptions)).filter(obs => obs !== undefined));
    }))]).subscribe({
      next: resp => {
        const statsMap = (0, _lodash.cloneDeep)(resp[0]);
        const fieldBatches = resp[1];
        if (Array.isArray(fieldBatches)) {
          fieldBatches.forEach(f => {
            if (Array.isArray(f) && f.length === 1 && Array.isArray(f[0].fields) && f[0].fields.length > 0) {
              statsMap.set(f[0].fields[0], f[0]);
            }
          });
          setFieldStats(statsMap);
          setFetchState({
            loaded: statsMap.size / pageOfConfigs.length * 100,
            isRunning: true
          });
        }
      },
      error: onError,
      complete: onComplete
    });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [data.search, toasts, fieldStatsParams, dataVisualizerListState.pageSize, dataVisualizerListState.pageIndex, dataVisualizerListState.sortDirection, dataVisualizerListState.sortField, samplingProbability]);
  const cancelFetch = (0, _react.useCallback)(() => {
    var _searchSubscription$$2, _retries$$current2;
    (_searchSubscription$$2 = searchSubscription$.current) === null || _searchSubscription$$2 === void 0 ? void 0 : _searchSubscription$$2.unsubscribe();
    searchSubscription$.current = undefined;
    (_retries$$current2 = retries$.current) === null || _retries$$current2 === void 0 ? void 0 : _retries$$current2.unsubscribe();
    retries$.current = undefined;
    abortCtrl.current.abort();
    setFetchState({
      isRunning: false
    });
  }, []);

  // auto-update
  (0, _react.useEffect)(() => {
    startFetch();
  }, [startFetch]);
  (0, _react.useEffect)(() => {
    return cancelFetch;
  }, [cancelFetch]);
  return {
    progress: fetchState,
    fieldStats,
    startFetch,
    cancelFetch
  };
}