"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installWithTimeout = exports.InstallShutdownError = void 0;
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const INSTALLATION_TIMEOUT = 20 * 60 * 1000; // 20 minutes

class InstallShutdownError extends Error {
  constructor() {
    super('Server is stopping; must stop all async operations');
    Object.setPrototypeOf(this, InstallShutdownError.prototype);
  }
}
exports.InstallShutdownError = InstallShutdownError;
const installWithTimeout = async ({
  description,
  installFn,
  pluginStop$,
  timeoutMs = INSTALLATION_TIMEOUT
}) => {
  try {
    let timeoutId;
    const install = async () => {
      await installFn();
      if (timeoutId) {
        clearTimeout(timeoutId);
      }
    };
    const throwTimeoutException = () => {
      return new Promise((_, reject) => {
        timeoutId = setTimeout(() => {
          const msg = `Timeout: it took more than ${timeoutMs}ms`;
          reject(new Error(msg));
        }, timeoutMs);
        (0, _rxjs.firstValueFrom)(pluginStop$).then(() => {
          clearTimeout(timeoutId);
          reject(new InstallShutdownError());
        });
      });
    };
    await Promise.race([install(), throwTimeoutException()]);
  } catch (e) {
    if (e instanceof InstallShutdownError) {
      throw e;
    } else {
      const reason = (e === null || e === void 0 ? void 0 : e.message) || 'Unknown reason';
      throw new Error(`Failure during installation${description ? ` of ${description}` : ''}. ${reason}`);
    }
  }
};
exports.installWithTimeout = installWithTimeout;