"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateStatus$ = void 0;
var _rxjs = require("rxjs");
var _coreStatusCommon = require("@kbn/core-status-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const calculateStatus$ = (rawMigratorStatus$, elasticsearchStatus$) => {
  const migratorStatus$ = rawMigratorStatus$.pipe((0, _rxjs.map)(migratorStatusToServiceStatus), (0, _rxjs.startWith)({
    level: _coreStatusCommon.ServiceStatusLevels.unavailable,
    summary: `SavedObjects service is waiting to start migrations`
  }));
  return (0, _rxjs.combineLatest)([elasticsearchStatus$, migratorStatus$]).pipe((0, _rxjs.map)(([esStatus, migratorStatus]) => {
    if (esStatus.level >= _coreStatusCommon.ServiceStatusLevels.unavailable) {
      return {
        level: _coreStatusCommon.ServiceStatusLevels.unavailable,
        summary: `SavedObjects service is not available without a healthy Elasticearch connection`
      };
    } else if (migratorStatus.level === _coreStatusCommon.ServiceStatusLevels.unavailable) {
      return migratorStatus;
    } else if (esStatus.level === _coreStatusCommon.ServiceStatusLevels.degraded) {
      return {
        level: esStatus.level,
        summary: `SavedObjects service is degraded due to Elasticsearch: [${esStatus.summary}]`
      };
    } else {
      return migratorStatus;
    }
  }));
};
exports.calculateStatus$ = calculateStatus$;
const migratorStatusToServiceStatus = migrationStatus => {
  if (migrationStatus.status === 'waiting_to_start') {
    return {
      level: _coreStatusCommon.ServiceStatusLevels.unavailable,
      summary: `SavedObjects service is waiting to start migrations`
    };
  } else if (migrationStatus.status === 'waiting_for_other_nodes') {
    return {
      level: _coreStatusCommon.ServiceStatusLevels.unavailable,
      summary: `SavedObjects service is waiting for other nodes to complete the migration`,
      detail: `If no other Kibana instance is attempting ` + `migrations, you can get past this message by deleting index ${migrationStatus.waitingIndex} and ` + `restarting Kibana.`
    };
  } else if (migrationStatus.status === 'running') {
    return {
      level: _coreStatusCommon.ServiceStatusLevels.unavailable,
      summary: `SavedObjects service is running migrations`
    };
  }
  const statusCounts = {
    migrated: 0,
    skipped: 0
  };
  if (migrationStatus.result) {
    migrationStatus.result.forEach(({
      status
    }) => {
      var _statusCounts$status;
      statusCounts[status] = ((_statusCounts$status = statusCounts[status]) !== null && _statusCounts$status !== void 0 ? _statusCounts$status : 0) + 1;
    });
  }
  return {
    level: _coreStatusCommon.ServiceStatusLevels.available,
    summary: `SavedObjects service has completed migrations and is available`,
    meta: {
      migratedIndices: statusCounts
    }
  };
};