"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PluginsSystem = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _std = require("@kbn/std");
var _coreBaseCommon = require("@kbn/core-base-common");
var _plugin_context = require("./plugin_context");
var _plugin_contract_resolver = require("./plugin_contract_resolver");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const Sec = 1000;

/** @internal */
class PluginsSystem {
  constructor(coreContext, type) {
    (0, _defineProperty2.default)(this, "runtimeResolver", new _plugin_contract_resolver.RuntimePluginContractResolver());
    (0, _defineProperty2.default)(this, "plugins", new Map());
    (0, _defineProperty2.default)(this, "log", void 0);
    // `satup`, the past-tense version of the noun `setup`.
    (0, _defineProperty2.default)(this, "satupPlugins", []);
    (0, _defineProperty2.default)(this, "sortedPluginNames", void 0);
    this.coreContext = coreContext;
    this.type = type;
    this.log = coreContext.logger.get('plugins-system', this.type);
  }
  addPlugin(plugin) {
    if (plugin.manifest.type !== this.type) {
      throw new Error(`Cannot add plugin with type "${plugin.manifest.type}" to plugin system with type "${this.type}".`);
    }
    this.plugins.set(plugin.name, plugin);

    // clear sorted plugin name cache on addition
    this.sortedPluginNames = undefined;
  }
  getPlugins() {
    return [...this.plugins.values()];
  }

  /**
   * @returns a Map of each plugin and an Array of its available dependencies
   * @internal
   */
  getPluginDependencies() {
    const asNames = new Map();
    const asOpaqueIds = new Map();
    for (const pluginName of this.getTopologicallySortedPluginNames()) {
      const plugin = this.plugins.get(pluginName);
      const dependencies = [...new Set([...plugin.requiredPlugins, ...plugin.optionalPlugins.filter(optPlugin => this.plugins.has(optPlugin))])];
      asNames.set(plugin.name, dependencies.map(depId => this.plugins.get(depId).name));
      asOpaqueIds.set(plugin.opaqueId, dependencies.map(depId => this.plugins.get(depId).opaqueId));
    }
    return {
      asNames,
      asOpaqueIds
    };
  }
  async setupPlugins(deps) {
    const contracts = new Map();
    if (this.plugins.size === 0) {
      return contracts;
    }
    const runtimeDependencies = buildPluginRuntimeDependencyMap(this.plugins);
    this.runtimeResolver.setDependencyMap(runtimeDependencies);
    const sortedPlugins = new Map([...this.getTopologicallySortedPluginNames()].map(pluginName => [pluginName, this.plugins.get(pluginName)]).filter(([pluginName, plugin]) => plugin.includesServerPlugin));
    this.log.info(`Setting up [${sortedPlugins.size}] plugins: [${[...sortedPlugins.keys()].join(',')}]`);
    for (const [pluginName, plugin] of sortedPlugins) {
      this.log.debug(`Setting up plugin "${pluginName}"...`);
      const pluginDeps = new Set([...plugin.requiredPlugins, ...plugin.optionalPlugins]);
      const pluginDepContracts = Array.from(pluginDeps).reduce((depContracts, dependencyName) => {
        // Only set if present. Could be absent if plugin does not have server-side code or is a
        // missing optional dependency.
        if (contracts.has(dependencyName)) {
          depContracts[dependencyName] = contracts.get(dependencyName);
        }
        return depContracts;
      }, {});
      let pluginSetupContext;
      if (this.type === _coreBaseCommon.PluginType.preboot) {
        pluginSetupContext = (0, _plugin_context.createPluginPrebootSetupContext)({
          deps: deps,
          plugin
        });
      } else {
        pluginSetupContext = (0, _plugin_context.createPluginSetupContext)({
          deps: deps,
          plugin,
          runtimeResolver: this.runtimeResolver
        });
      }
      await plugin.init();
      let contract;
      const contractOrPromise = plugin.setup(pluginSetupContext, pluginDepContracts);
      if ((0, _std.isPromise)(contractOrPromise)) {
        if (this.coreContext.env.mode.dev) {
          this.log.warn(`Plugin ${pluginName} is using asynchronous setup lifecycle. Asynchronous plugins support will be removed in a later version.`);
        }
        const contractMaybe = await (0, _std.withTimeout)({
          promise: contractOrPromise,
          timeoutMs: 10 * Sec
        });
        if (contractMaybe.timedout) {
          throw new Error(`Setup lifecycle of "${pluginName}" plugin wasn't completed in 10sec. Consider disabling the plugin and re-start.`);
        } else {
          contract = contractMaybe.value;
        }
      } else {
        contract = contractOrPromise;
      }
      contracts.set(pluginName, contract);
      this.satupPlugins.push(pluginName);
    }
    this.runtimeResolver.resolveSetupRequests(contracts);
    return contracts;
  }
  async startPlugins(deps) {
    if (this.type === _coreBaseCommon.PluginType.preboot) {
      throw new Error('Preboot plugins cannot be started.');
    }
    const contracts = new Map();
    if (this.satupPlugins.length === 0) {
      return contracts;
    }
    this.log.info(`Starting [${this.satupPlugins.length}] plugins: [${[...this.satupPlugins]}]`);
    for (const pluginName of this.satupPlugins) {
      this.log.debug(`Starting plugin "${pluginName}"...`);
      const plugin = this.plugins.get(pluginName);
      const pluginDeps = new Set([...plugin.requiredPlugins, ...plugin.optionalPlugins]);
      const pluginDepContracts = Array.from(pluginDeps).reduce((depContracts, dependencyName) => {
        // Only set if present. Could be absent if plugin does not have server-side code or is a
        // missing optional dependency.
        if (contracts.has(dependencyName)) {
          depContracts[dependencyName] = contracts.get(dependencyName);
        }
        return depContracts;
      }, {});
      let contract;
      const contractOrPromise = plugin.start((0, _plugin_context.createPluginStartContext)({
        deps,
        plugin,
        runtimeResolver: this.runtimeResolver
      }), pluginDepContracts);
      if ((0, _std.isPromise)(contractOrPromise)) {
        if (this.coreContext.env.mode.dev) {
          this.log.warn(`Plugin ${pluginName} is using asynchronous start lifecycle. Asynchronous plugins support will be removed in a later version.`);
        }
        const contractMaybe = await (0, _std.withTimeout)({
          promise: contractOrPromise,
          timeoutMs: 10 * Sec
        });
        if (contractMaybe.timedout) {
          throw new Error(`Start lifecycle of "${pluginName}" plugin wasn't completed in 10sec. Consider disabling the plugin and re-start.`);
        } else {
          contract = contractMaybe.value;
        }
      } else {
        contract = contractOrPromise;
      }
      contracts.set(pluginName, contract);
    }
    this.runtimeResolver.resolveStartRequests(contracts);
    return contracts;
  }
  async stopPlugins() {
    if (this.plugins.size === 0 || this.satupPlugins.length === 0) {
      return;
    }
    this.log.info(`Stopping all plugins.`);
    const reverseDependencyMap = buildReverseDependencyMap(this.plugins);
    const pluginStopPromiseMap = new Map();
    for (let i = this.satupPlugins.length - 1; i > -1; i--) {
      const pluginName = this.satupPlugins[i];
      const plugin = this.plugins.get(pluginName);
      const pluginDependant = reverseDependencyMap.get(pluginName);
      const dependantPromises = pluginDependant.map(dependantName => pluginStopPromiseMap.get(dependantName));

      // Stop plugin as soon as all the dependant plugins are stopped.
      const pluginStopPromise = Promise.all(dependantPromises).then(async () => {
        this.log.debug(`Stopping plugin "${pluginName}"...`);
        try {
          const resultMaybe = await (0, _std.withTimeout)({
            promise: plugin.stop(),
            timeoutMs: 15 * Sec
          });
          if (resultMaybe !== null && resultMaybe !== void 0 && resultMaybe.timedout) {
            this.log.warn(`"${pluginName}" plugin didn't stop in 15sec., move on to the next.`);
          }
        } catch (e) {
          this.log.warn(`"${pluginName}" thrown during stop: ${e}`);
        }
      });
      pluginStopPromiseMap.set(pluginName, pluginStopPromise);
    }
    await Promise.allSettled(pluginStopPromiseMap.values());
    this.log.info(`All plugins stopped.`);
  }

  /**
   * Get a Map of all discovered UI plugins in topological order.
   */
  uiPlugins() {
    const uiPluginNames = [...this.getTopologicallySortedPluginNames().keys()].filter(pluginName => this.plugins.get(pluginName).includesUiPlugin);
    const filterUiPlugins = pluginName => uiPluginNames.includes(pluginName);
    const publicPlugins = new Map(uiPluginNames.map(pluginName => {
      const plugin = this.plugins.get(pluginName);
      return [pluginName, {
        id: pluginName,
        type: plugin.manifest.type,
        configPath: plugin.manifest.configPath,
        requiredPlugins: plugin.manifest.requiredPlugins.filter(filterUiPlugins),
        optionalPlugins: plugin.manifest.optionalPlugins.filter(filterUiPlugins),
        runtimePluginDependencies: plugin.manifest.runtimePluginDependencies.filter(filterUiPlugins),
        requiredBundles: plugin.manifest.requiredBundles,
        enabledOnAnonymousPages: plugin.manifest.enabledOnAnonymousPages
      }];
    }));
    return publicPlugins;
  }
  getTopologicallySortedPluginNames() {
    if (!this.sortedPluginNames) {
      this.sortedPluginNames = getTopologicallySortedPluginNames(this.plugins);
    }
    return this.sortedPluginNames;
  }
}

/**
 * Gets topologically sorted plugin names that are registered with the plugin system.
 * Ordering is possible if and only if the plugins graph has no directed cycles,
 * that is, if it is a directed acyclic graph (DAG). If plugins cannot be ordered
 * an error is thrown.
 *
 * Uses Kahn's Algorithm to sort the graph.
 */
exports.PluginsSystem = PluginsSystem;
const getTopologicallySortedPluginNames = plugins => {
  // We clone plugins so we can remove handled nodes while we perform the
  // topological ordering. If the cloned graph is _not_ empty at the end, we
  // know we were not able to topologically order the graph. We exclude optional
  // dependencies that are not present in the plugins graph.
  const pluginsDependenciesGraph = new Map([...plugins.entries()].map(([pluginName, plugin]) => {
    return [pluginName, new Set([...plugin.requiredPlugins, ...plugin.optionalPlugins.filter(dependency => plugins.has(dependency))])];
  }));

  // First, find a list of "start nodes" which have no outgoing edges. At least
  // one such node must exist in a non-empty acyclic graph.
  const pluginsWithAllDependenciesSorted = [...pluginsDependenciesGraph.keys()].filter(pluginName => pluginsDependenciesGraph.get(pluginName).size === 0);
  const sortedPluginNames = new Set();
  while (pluginsWithAllDependenciesSorted.length > 0) {
    const sortedPluginName = pluginsWithAllDependenciesSorted.pop();

    // We know this plugin has all its dependencies sorted, so we can remove it
    // and include into the final result.
    pluginsDependenciesGraph.delete(sortedPluginName);
    sortedPluginNames.add(sortedPluginName);

    // Go through the rest of the plugins and remove `sortedPluginName` from their
    // unsorted dependencies.
    for (const [pluginName, dependencies] of pluginsDependenciesGraph) {
      // If we managed delete `sortedPluginName` from dependencies let's check
      // whether it was the last one and we can mark plugin as sorted.
      if (dependencies.delete(sortedPluginName) && dependencies.size === 0) {
        pluginsWithAllDependenciesSorted.push(pluginName);
      }
    }
  }
  if (pluginsDependenciesGraph.size > 0) {
    const edgesLeft = JSON.stringify([...pluginsDependenciesGraph.keys()]);
    throw new Error(`Topological ordering of plugins did not complete, these plugins have cyclic or missing dependencies: ${edgesLeft}`);
  }
  return sortedPluginNames;
};
const buildReverseDependencyMap = pluginMap => {
  const reverseMap = new Map();
  for (const pluginName of pluginMap.keys()) {
    reverseMap.set(pluginName, []);
  }
  for (const [pluginName, pluginWrapper] of pluginMap.entries()) {
    const allDependencies = [...pluginWrapper.requiredPlugins, ...pluginWrapper.optionalPlugins];
    for (const dependency of allDependencies) {
      // necessary to evict non-present optional dependency
      if (pluginMap.has(dependency)) {
        reverseMap.get(dependency).push(pluginName);
      }
    }
    reverseMap.set(pluginName, []);
  }
  return reverseMap;
};
const buildPluginRuntimeDependencyMap = pluginMap => {
  const runtimeDependencies = new Map();
  for (const [pluginName, pluginWrapper] of pluginMap.entries()) {
    const pluginRuntimeDeps = new Set([...pluginWrapper.optionalPlugins, ...pluginWrapper.requiredPlugins, ...pluginWrapper.runtimePluginDependencies]);
    runtimeDependencies.set(pluginName, pluginRuntimeDeps);
  }
  return runtimeDependencies;
};