"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MonacoEditorActionsProvider = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _monaco = require("@kbn/monaco");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../../common/constants");
var _services = require("../../../../services");
var _send_request = require("../../../hooks/use_send_current_request/send_request");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const selectedRequestsClass = 'console__monaco_editor__selectedRequests';
class MonacoEditorActionsProvider {
  constructor(editor, setEditorActionsCss) {
    (0, _defineProperty2.default)(this, "parsedRequestsProvider", void 0);
    (0, _defineProperty2.default)(this, "highlightedLines", void 0);
    this.editor = editor;
    this.setEditorActionsCss = setEditorActionsCss;
    this.parsedRequestsProvider = (0, _monaco.getParsedRequestsProvider)(this.editor.getModel());
    this.highlightedLines = this.editor.createDecorationsCollection();
    this.editor.focus();
    const debouncedHighlightRequests = (0, _lodash.debounce)(() => this.highlightRequests(), 200, {
      leading: true
    });
    debouncedHighlightRequests();

    // init all listeners
    editor.onDidChangeCursorPosition(async event => {
      await debouncedHighlightRequests();
    });
    editor.onDidScrollChange(async event => {
      await debouncedHighlightRequests();
    });
    editor.onDidChangeCursorSelection(async event => {
      await debouncedHighlightRequests();
    });
    editor.onDidContentSizeChange(async event => {
      await debouncedHighlightRequests();
    });
  }
  updateEditorActions(lineNumber) {
    // if no request is currently selected, hide the actions buttons
    if (!lineNumber) {
      this.setEditorActionsCss({
        visibility: 'hidden'
      });
    } else {
      // if a request is selected, the actions buttons are placed at lineNumberOffset - scrollOffset
      const offset = this.editor.getTopForLineNumber(lineNumber) - this.editor.getScrollTop();
      this.setEditorActionsCss({
        visibility: 'visible',
        top: offset
      });
    }
  }
  async highlightRequests() {
    // get the requests in the selected range
    const {
      range: selectedRange,
      parsedRequests
    } = await this.getSelectedParsedRequestsAndRange();
    // if any requests are selected, highlight the lines and update the position of actions buttons
    if (parsedRequests.length > 0) {
      const selectedRequestStartLine = selectedRange.startLineNumber;
      // display the actions buttons on the 1st line of the 1st selected request
      this.updateEditorActions(selectedRequestStartLine);
      this.highlightedLines.set([{
        range: selectedRange,
        options: {
          isWholeLine: true,
          className: selectedRequestsClass
        }
      }]);
    } else {
      // if no requests are selected, hide actions buttons and remove highlighted lines
      this.updateEditorActions();
      this.highlightedLines.clear();
    }
  }
  async getSelectedParsedRequestsAndRange() {
    const model = this.editor.getModel();
    const selection = this.editor.getSelection();
    if (!model || !selection) {
      return Promise.resolve({
        parsedRequests: [],
        range: selection !== null && selection !== void 0 ? selection : new _monaco.monaco.Range(1, 1, 1, 1)
      });
    }
    const {
      startLineNumber,
      endLineNumber
    } = selection;
    const parsedRequests = await this.parsedRequestsProvider.getRequests();
    const selectedRequests = [];
    let selectionStartLine = startLineNumber;
    let selectionEndLine = endLineNumber;
    for (const parsedRequest of parsedRequests) {
      const {
        startOffset: requestStart,
        endOffset: requestEnd
      } = parsedRequest;
      const {
        lineNumber: requestStartLine
      } = model.getPositionAt(requestStart);
      let {
        lineNumber: requestEndLine
      } = model.getPositionAt(requestEnd);
      const requestEndLineContent = model.getLineContent(requestEndLine);

      // sometimes the parser includes a trailing empty line into the request
      if (requestEndLineContent.trim().length < 1) {
        requestEndLine = requestEndLine - 1;
      }
      if (requestStartLine > endLineNumber) {
        // request is past the selection, no need to check further requests
        break;
      }
      if (requestEndLine < startLineNumber) {
        // request is before the selection, do nothing
      } else {
        // request is selected
        selectedRequests.push(parsedRequest);
        // expand the start of the selection to the request start
        if (selectionStartLine > requestStartLine) {
          selectionStartLine = requestStartLine;
        }
        // expand the end of the selection to the request end
        if (selectionEndLine < requestEndLine) {
          selectionEndLine = requestEndLine;
        }
      }
    }
    return {
      parsedRequests: selectedRequests,
      // the expanded selected range goes from the 1st char of the start line of the 1st request
      // to the last char of the last line of the last request
      range: new _monaco.monaco.Range(selectionStartLine, 1, selectionEndLine, model.getLineMaxColumn(selectionEndLine))
    };
  }
  async getRequests() {
    const {
      parsedRequests
    } = await this.getSelectedParsedRequestsAndRange();
    const stringifiedRequests = parsedRequests.map(parsedRequest => (0, _utils.stringifyRequest)(parsedRequest));
    // get variables values
    const variables = (0, _services.getStorage)().get(_services.StorageKeys.VARIABLES, _constants.DEFAULT_VARIABLES);
    return stringifiedRequests.map(request => (0, _utils.replaceRequestVariables)(request, variables));
  }
  async getCurl(elasticsearchBaseUrl) {
    const requests = await this.getRequests();
    const curlRequests = requests.map(request => (0, _utils.getCurlRequest)(request, elasticsearchBaseUrl));
    return curlRequests.join('\n');
  }
  async sendRequests(toasts, dispatch, trackUiMetric, http) {
    try {
      const requests = await this.getRequests();
      if (!requests.length) {
        toasts.add(_i18n.i18n.translate('console.notification.error.noRequestSelectedTitle', {
          defaultMessage: 'No request selected. Select a request by placing the cursor inside it.'
        }));
        return;
      }
      dispatch({
        type: 'sendRequest',
        payload: undefined
      });

      // track the requests
      setTimeout(() => (0, _utils.trackSentRequests)(requests, trackUiMetric), 0);
      const results = await (0, _send_request.sendRequest)({
        http,
        requests
      });

      // TODO save to history
      // TODO restart autocomplete polling
      dispatch({
        type: 'requestSuccess',
        payload: {
          data: results
        }
      });
    } catch (e) {
      if (e !== null && e !== void 0 && e.response) {
        dispatch({
          type: 'requestFail',
          payload: e
        });
      } else {
        dispatch({
          type: 'requestFail',
          payload: undefined
        });
        toasts.addError(e, {
          title: _i18n.i18n.translate('console.notification.error.unknownErrorTitle', {
            defaultMessage: 'Unknown Request Error'
          })
        });
      }
    }
  }
}
exports.MonacoEditorActionsProvider = MonacoEditorActionsProvider;