"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.onPackagePolicyPostCreateCallback = exports.getInstalledPolicyTemplates = exports.getCloudDefendPackagePolicies = exports.getCloudDefendAgentPolicies = exports.getAgentStatusesByAgentPolicies = exports.PACKAGE_POLICY_SAVED_OBJECT_TYPE = void 0;
var _lodash = require("lodash");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PACKAGE_POLICY_SAVED_OBJECT_TYPE = exports.PACKAGE_POLICY_SAVED_OBJECT_TYPE = 'ingest-package-policies';
const isFleetMissingAgentHttpError = error => error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 404;
const isPolicyTemplate = input => input === _constants.INPUT_CONTROL;
const getPackageNameQuery = (packageName, benchmarkFilter) => {
  const integrationNameQuery = `${PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${packageName}`;
  const kquery = benchmarkFilter ? `${integrationNameQuery} AND ${PACKAGE_POLICY_SAVED_OBJECT_TYPE}.name: *${benchmarkFilter}*` : integrationNameQuery;
  return kquery;
};
const getAgentStatusesByAgentPolicies = async (agentService, agentPolicies, logger) => {
  if (!(agentPolicies !== null && agentPolicies !== void 0 && agentPolicies.length)) return {};
  const internalAgentService = agentService.asInternalUser;
  const result = {};
  try {
    for (const agentPolicy of agentPolicies) {
      result[agentPolicy.id] = await internalAgentService.getAgentStatusForAgentPolicy(agentPolicy.id);
    }
  } catch (error) {
    if (isFleetMissingAgentHttpError(error)) {
      logger.debug('failed to get agent status for agent policy');
    } else {
      throw error;
    }
  }
  return result;
};
exports.getAgentStatusesByAgentPolicies = getAgentStatusesByAgentPolicies;
const onPackagePolicyPostCreateCallback = async (logger, packagePolicy, savedObjectsClient) => {
  return addDataViewToAllSpaces(savedObjectsClient);
};
exports.onPackagePolicyPostCreateCallback = onPackagePolicyPostCreateCallback;
async function addDataViewToAllSpaces(savedObjectsClient) {
  const cloudDefendDataViews = await savedObjectsClient.find({
    type: 'index-pattern',
    fields: ['title'],
    search: _constants.INTEGRATION_PACKAGE_NAME + '*',
    searchFields: ['title'],
    perPage: 100
  });
  cloudDefendDataViews.saved_objects.forEach(dataView => {
    savedObjectsClient.updateObjectsSpaces([{
      id: dataView.id,
      type: 'index-pattern'
    }], ['*'], []);
  });
}
const getCloudDefendAgentPolicies = async (soClient, packagePolicies, agentPolicyService) => agentPolicyService.getByIds(soClient, (0, _lodash.uniq)((0, _lodash.map)(packagePolicies, 'policy_id')), {
  withPackagePolicies: true,
  ignoreMissing: true
});
exports.getCloudDefendAgentPolicies = getCloudDefendAgentPolicies;
const getCloudDefendPackagePolicies = (soClient, packagePolicyService, packageName, queryParams) => {
  var _queryParams$sort_fie;
  const sortField = (_queryParams$sort_fie = queryParams.sort_field) === null || _queryParams$sort_fie === void 0 ? void 0 : _queryParams$sort_fie.replaceAll(_constants.POLICIES_PACKAGE_POLICY_PREFIX, '');
  return packagePolicyService.list(soClient, {
    kuery: getPackageNameQuery(packageName, queryParams.policy_name),
    page: queryParams.page,
    perPage: queryParams.per_page,
    sortField,
    sortOrder: queryParams.sort_order
  });
};
exports.getCloudDefendPackagePolicies = getCloudDefendPackagePolicies;
const getInstalledPolicyTemplates = async (packagePolicyClient, soClient) => {
  try {
    // getting all installed cloud_defend package policies
    const queryResult = await packagePolicyClient.list(soClient, {
      kuery: _constants.CLOUD_DEFEND_FLEET_PACKAGE_KUERY,
      perPage: 1000
    });

    // getting installed policy templates
    const enabledPolicyTemplates = queryResult.items.map(policy => {
      var _policy$inputs$find;
      return (_policy$inputs$find = policy.inputs.find(input => input.enabled)) === null || _policy$inputs$find === void 0 ? void 0 : _policy$inputs$find.policy_template;
    }).filter(isPolicyTemplate);

    // removing duplicates
    return [...new Set(enabledPolicyTemplates)];
  } catch (e) {
    return [];
  }
};
exports.getInstalledPolicyTemplates = getInstalledPolicyTemplates;