"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CaseConnectorRetryService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _cases_connector_error = require("./cases_connector_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CaseConnectorRetryService {
  constructor(logger, backOffFactory, maxAttempts = 10) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "maxAttempts", void 0);
    /**
     * 409 - Conflict
     * 429 - Too Many Requests
     * 503 - ES Unavailable
     *
     * Full list of errors: packages/core/saved-objects/core-saved-objects-server/src/saved_objects_error_helpers.ts
     */
    (0, _defineProperty2.default)(this, "RETRY_ERROR_STATUS_CODES", [409, 429, 503]);
    (0, _defineProperty2.default)(this, "backOffStrategy", void 0);
    (0, _defineProperty2.default)(this, "timer", null);
    (0, _defineProperty2.default)(this, "attempt", 0);
    this.logger = logger;
    this.backOffStrategy = backOffFactory.create();
    this.maxAttempts = maxAttempts;
  }
  async retryWithBackoff(cb) {
    try {
      this.logger.debug(`[CasesConnector][retryWithBackoff] Running case connector. Attempt: ${this.attempt}`, {
        labels: {
          attempt: this.attempt
        },
        tags: ['case-connector:retry-start']
      });
      const res = await cb();
      this.logger.debug(`[CasesConnector][retryWithBackoff] Case connector run successfully after ${this.attempt} attempts`, {
        labels: {
          attempt: this.attempt
        },
        tags: ['case-connector:retry-success']
      });
      return res;
    } catch (error) {
      if (this.shouldRetry() && this.isRetryableError(error)) {
        this.stop();
        this.attempt++;
        await this.delay();
        this.logger.warn(`[CaseConnector] Case connector failed with status code ${error.statusCode}. Attempt for retry: ${this.attempt}`);
        return this.retryWithBackoff(cb);
      }
      throw error;
    } finally {
      this.logger.debug(`[CasesConnector][retryWithBackoff] Case connector run ended after ${this.attempt} attempts`, {
        labels: {
          attempt: this.attempt
        },
        tags: ['case-connector:retry-end']
      });
    }
  }
  shouldRetry() {
    return this.attempt < this.maxAttempts;
  }
  isRetryableError(error) {
    if (error instanceof _cases_connector_error.CasesConnectorError && this.RETRY_ERROR_STATUS_CODES.includes(error.statusCode)) {
      return true;
    }
    this.logger.debug(`[CasesConnector][isRetryableError] Error is not retryable`, {
      tags: ['case-connector:retry-error']
    });
    return false;
  }
  async delay() {
    const ms = this.backOffStrategy.nextBackOff();
    return new Promise(resolve => {
      this.timer = setTimeout(resolve, ms);
    });
  }
  stop() {
    if (this.timer !== null) {
      clearTimeout(this.timer);
      this.timer = null;
    }
  }
}
exports.CaseConnectorRetryService = CaseConnectorRetryService;