"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkCreate = void 0;
var _server = require("@kbn/core/server");
var _api = require("../../../common/types/api");
var _runtime_types = require("../../common/runtime_types");
var _models = require("../../common/models");
var _error = require("../../common/error");
var _utils = require("../utils");
var _authorization = require("../../authorization");
var _validators = require("./validators");
var _validators2 = require("../../common/validators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkCreate = async (args, clientArgs) => {
  const {
    attachments,
    caseId
  } = args;
  const {
    logger,
    authorization,
    externalReferenceAttachmentTypeRegistry,
    persistableStateAttachmentTypeRegistry,
    services: {
      userActionService
    }
  } = clientArgs;
  try {
    (0, _runtime_types.decodeWithExcessOrThrow)(_api.BulkCreateAttachmentsRequestRt)(attachments);
    await (0, _validators2.validateMaxUserActions)({
      caseId,
      userActionService,
      userActionsToAdd: attachments.length
    });
    attachments.forEach(attachment => {
      (0, _utils.decodeCommentRequest)(attachment, externalReferenceAttachmentTypeRegistry);
      (0, _validators.validateRegisteredAttachments)({
        query: attachment,
        persistableStateAttachmentTypeRegistry,
        externalReferenceAttachmentTypeRegistry
      });
    });
    const [attachmentsWithIds, entities] = attachments.reduce(([a, e], attachment) => {
      const savedObjectID = _server.SavedObjectsUtils.generateId();
      return [[...a, {
        id: savedObjectID,
        ...attachment
      }], [...e, {
        owner: attachment.owner,
        id: savedObjectID
      }]];
    }, [[], []]);
    await authorization.ensureAuthorized({
      operation: _authorization.Operations.bulkCreateAttachments,
      entities
    });
    const model = await _models.CaseCommentModel.create(caseId, clientArgs);
    const updatedModel = await model.bulkCreate({
      attachments: attachmentsWithIds
    });
    return await updatedModel.encodeWithComments();
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed while bulk creating attachment to case id: ${caseId} error: ${error}`,
      error,
      logger
    });
  }
};
exports.bulkCreate = bulkCreate;