"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _lodash = require("lodash");
var _uuid = require("uuid");
var _server = require("@kbn/task-manager-plugin/server");
var _server2 = require("@kbn/event-log-plugin/server");
var _task_running = require("@kbn/task-manager-plugin/server/task_running");
var _execution_handler = require("./execution_handler");
var _get_executor_services = require("./get_executor_services");
var _lib = require("../lib");
var _types = require("../types");
var _result_type = require("../lib/result_type");
var _alert_task_instance = require("./alert_task_instance");
var _is_alerting_error = require("../lib/is_alerting_error");
var _saved_objects = require("../saved_objects");
var _common = require("../../common");
var _errors = require("../lib/errors");
var _monitoring = require("../monitoring");
var _rule_run_metrics_store = require("../lib/rule_run_metrics_store");
var _alerting_event_logger = require("../lib/alerting_event_logger/alerting_event_logger");
var _rule_loader = require("./rule_loader");
var _task_runner_timer = require("./task_runner_timer");
var _rule_monitoring_service = require("../monitoring/rule_monitoring_service");
var _last_run_status = require("../lib/last_run_status");
var _running_handler = require("./running_handler");
var _rule_result_service = require("../monitoring/rule_result_service");
var _get_maintenance_windows = require("./get_maintenance_windows");
var _rule_type_runner = require("./rule_type_runner");
var _alerts_client = require("../alerts_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FALLBACK_RETRY_INTERVAL = '5m';
const CONNECTIVITY_RETRY_INTERVAL = '5m';
class TaskRunner {
  constructor({
    ruleType,
    taskInstance,
    context,
    inMemoryMetrics
  }) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskInstance", void 0);
    (0, _defineProperty2.default)(this, "ruleConsumer", void 0);
    (0, _defineProperty2.default)(this, "ruleType", void 0);
    (0, _defineProperty2.default)(this, "executionId", void 0);
    (0, _defineProperty2.default)(this, "ruleTypeRegistry", void 0);
    (0, _defineProperty2.default)(this, "inMemoryMetrics", void 0);
    (0, _defineProperty2.default)(this, "timer", void 0);
    (0, _defineProperty2.default)(this, "alertingEventLogger", void 0);
    (0, _defineProperty2.default)(this, "usageCounter", void 0);
    (0, _defineProperty2.default)(this, "searchAbortController", void 0);
    (0, _defineProperty2.default)(this, "cancelled", void 0);
    (0, _defineProperty2.default)(this, "stackTraceLog", void 0);
    (0, _defineProperty2.default)(this, "ruleMonitoring", void 0);
    (0, _defineProperty2.default)(this, "ruleRunning", void 0);
    (0, _defineProperty2.default)(this, "ruleResult", void 0);
    (0, _defineProperty2.default)(this, "maintenanceWindows", []);
    (0, _defineProperty2.default)(this, "maintenanceWindowsWithoutScopedQueryIds", []);
    (0, _defineProperty2.default)(this, "ruleTypeRunner", void 0);
    (0, _defineProperty2.default)(this, "runDate", new Date());
    this.context = context;
    const loggerId = ruleType.id.startsWith('.') ? ruleType.id.substring(1) : ruleType.id;
    this.logger = context.logger.get(loggerId);
    this.usageCounter = context.usageCounter;
    this.ruleType = ruleType;
    this.ruleConsumer = null;
    this.taskInstance = (0, _alert_task_instance.taskInstanceToAlertTaskInstance)(taskInstance);
    this.ruleTypeRegistry = context.ruleTypeRegistry;
    this.searchAbortController = new AbortController();
    this.cancelled = false;
    this.executionId = (0, _uuid.v4)();
    this.inMemoryMetrics = inMemoryMetrics;
    this.timer = new _task_runner_timer.TaskRunnerTimer({
      logger: this.logger
    });
    this.alertingEventLogger = new _alerting_event_logger.AlertingEventLogger(this.context.eventLogger);
    this.stackTraceLog = null;
    this.ruleMonitoring = new _rule_monitoring_service.RuleMonitoringService();
    this.ruleRunning = new _running_handler.RunningHandler(this.context.internalSavedObjectsRepository, this.logger, loggerId);
    this.ruleTypeRunner = new _rule_type_runner.RuleTypeRunner({
      context: this.context,
      logger: this.logger,
      timer: this.timer,
      ruleType: this.ruleType
    });
    this.ruleResult = new _rule_result_service.RuleResultService();
  }
  async updateRuleSavedObjectPostRun(ruleId, namespace, attributes) {
    const client = this.context.internalSavedObjectsRepository;
    try {
      // Future engineer -> Here we are just checking if we need to wait for
      // the update of the attribute `running` in the rule's saved object
      // and we are swallowing the error because we still want to move forward
      // with the update of our rule since we are putting back the running attribute
      // back to false
      await this.ruleRunning.waitFor();
      // eslint-disable-next-line no-empty
    } catch {}
    try {
      await (0, _saved_objects.partiallyUpdateRule)(client, ruleId, {
        ...attributes,
        running: false
      }, {
        ignore404: true,
        namespace,
        refresh: false
      });
    } catch (err) {
      this.logger.error(`error updating rule for ${this.ruleType.id}:${ruleId} ${err.message}`);
    }
  }
  shouldLogAndScheduleActionsForAlerts() {
    // if execution hasn't been cancelled, return true
    if (!this.cancelled) {
      return true;
    }

    // if execution has been cancelled, return true if EITHER alerting config or rule type indicate to proceed with scheduling actions
    return !this.context.cancelAlertsOnRuleTimeout || !this.ruleType.cancelAlertsOnRuleTimeout;
  }

  // Usage counter for telemetry
  // This keeps track of how many times action executions were skipped after rule
  // execution completed successfully after the execution timeout
  // This can occur when rule executors do not short circuit execution in response
  // to timeout
  countUsageOfActionExecutionAfterRuleCancellation() {
    if (this.cancelled && this.usageCounter) {
      if (this.context.cancelAlertsOnRuleTimeout && this.ruleType.cancelAlertsOnRuleTimeout) {
        // Increment usage counter for skipped actions
        this.usageCounter.incrementCounter({
          counterName: `alertsSkippedDueToRuleExecutionTimeout_${this.ruleType.id}`,
          incrementBy: 1
        });
      }
    }
  }
  async runRule({
    fakeRequest,
    rule,
    apiKey,
    validatedParams: params
  }) {
    if (_elasticApmNode.default.currentTransaction) {
      _elasticApmNode.default.currentTransaction.name = `Execute Alerting Rule: "${rule.name}"`;
      _elasticApmNode.default.currentTransaction.addLabels({
        alerting_rule_consumer: rule.consumer,
        alerting_rule_name: rule.name,
        alerting_rule_tags: rule.tags.join(', '),
        alerting_rule_type_id: rule.alertTypeId,
        alerting_rule_params: JSON.stringify(rule.params)
      });
    }
    const {
      params: {
        alertId: ruleId,
        spaceId
      },
      state: {
        previousStartedAt
      }
    } = this.taskInstance;
    const rulesSettingsClient = this.context.getRulesSettingsClientWithRequest(fakeRequest);
    const ruleRunMetricsStore = new _rule_run_metrics_store.RuleRunMetricsStore();
    const ruleLabel = `${this.ruleType.id}:${ruleId}: '${rule.name}'`;
    const ruleTypeRunnerContext = {
      alertingEventLogger: this.alertingEventLogger,
      flappingSettings: await rulesSettingsClient.flapping().get(),
      namespace: this.context.spaceIdToNamespace(spaceId),
      queryDelaySettings: await rulesSettingsClient.queryDelay().get(),
      ruleId,
      ruleLogPrefix: ruleLabel,
      ruleRunMetricsStore,
      spaceId
    };
    const alertsClient = await (0, _alerts_client.initializeAlertsClient)({
      alertsService: this.context.alertsService,
      context: ruleTypeRunnerContext,
      executionId: this.executionId,
      logger: this.logger,
      maxAlerts: this.context.maxAlerts,
      rule,
      ruleType: this.ruleType,
      taskInstance: this.taskInstance
    });
    const executorServices = await (0, _get_executor_services.getExecutorServices)({
      context: this.context,
      fakeRequest,
      abortController: this.searchAbortController,
      logger: this.logger,
      ruleMonitoringService: this.ruleMonitoring,
      ruleResultService: this.ruleResult,
      ruleData: {
        name: rule.name,
        alertTypeId: rule.alertTypeId,
        id: rule.id,
        spaceId
      },
      ruleTaskTimeout: this.ruleType.ruleTaskTimeout
    });
    const {
      state: updatedRuleTypeState,
      error,
      stackTrace
    } = await this.ruleTypeRunner.run({
      context: ruleTypeRunnerContext,
      alertsClient,
      executionId: this.executionId,
      executorServices,
      maintenanceWindows: this.maintenanceWindows,
      maintenanceWindowsWithoutScopedQueryIds: this.maintenanceWindowsWithoutScopedQueryIds,
      rule,
      startedAt: this.taskInstance.startedAt,
      state: this.taskInstance.state,
      validatedParams: params
    });

    // if there was an error, save the stack trace and throw
    if (error) {
      this.stackTraceLog = stackTrace !== null && stackTrace !== void 0 ? stackTrace : null;
      throw error;
    }
    const executionHandler = new _execution_handler.ExecutionHandler({
      rule,
      ruleType: this.ruleType,
      logger: this.logger,
      taskRunnerContext: this.context,
      taskInstance: this.taskInstance,
      ruleRunMetricsStore,
      apiKey,
      ruleConsumer: this.ruleConsumer,
      executionId: this.executionId,
      ruleLabel,
      previousStartedAt: previousStartedAt ? new Date(previousStartedAt) : null,
      alertingEventLogger: this.alertingEventLogger,
      actionsClient: await this.context.actionsPlugin.getActionsClientWithRequest(fakeRequest),
      alertsClient
    });
    let executionHandlerRunResult = {
      throttledSummaryActions: {}
    };
    await this.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.TriggerActions, async () => {
      if ((0, _lib.isRuleSnoozed)(rule)) {
        this.logger.debug(`no scheduling of actions for rule ${ruleLabel}: rule is snoozed.`);
      } else if (!this.shouldLogAndScheduleActionsForAlerts()) {
        this.logger.debug(`no scheduling of actions for rule ${ruleLabel}: rule execution has been cancelled.`);
        this.countUsageOfActionExecutionAfterRuleCancellation();
      } else {
        executionHandlerRunResult = await executionHandler.run({
          ...alertsClient.getProcessedAlerts('activeCurrent'),
          ...alertsClient.getProcessedAlerts('recoveredCurrent')
        });
      }
    });
    let alertsToReturn = {};
    let recoveredAlertsToReturn = {};

    // Only serialize alerts into task state if we're auto-recovering, otherwise
    // we don't need to keep this information around.
    if (this.ruleType.autoRecoverAlerts) {
      const {
        alertsToReturn: alerts,
        recoveredAlertsToReturn: recovered
      } = alertsClient.getAlertsToSerialize();
      alertsToReturn = alerts;
      recoveredAlertsToReturn = recovered;
    }
    return {
      metrics: ruleRunMetricsStore.getMetrics(),
      alertTypeState: updatedRuleTypeState || undefined,
      alertInstances: alertsToReturn,
      alertRecoveredInstances: recoveredAlertsToReturn,
      summaryActions: executionHandlerRunResult.throttledSummaryActions
    };
  }

  /**
   * Before we actually run the rule:
   * - start the RunningHandler
   * - initialize the event logger
   * - if rule data not loaded, load it
   * - set the current APM transaction info
   * - validate that rule type is enabled and params are valid
   * - initialize monitoring data
   * - clear expired snoozes
   */
  async prepareToRun() {
    return await this.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.PrepareRule, async () => {
      const {
        params: {
          alertId: ruleId,
          spaceId,
          consumer
        },
        startedAt
      } = this.taskInstance;

      // Initially use consumer as stored inside the task instance
      // This allows us to populate a consumer value for event log
      // `execute-start` events (which are indexed before the rule SO is read)
      // and in the event of decryption errors (where we cannot read the rule SO)
      // Because "consumer" is set when a rule is created, this value should be static
      // for the life of a rule but there may be edge cases where migrations cause
      // the consumer values to become out of sync.
      if (consumer) {
        this.ruleConsumer = consumer;
      }

      // Start the event logger so that something is logged in the
      // event that rule SO decryption fails.
      const namespace = this.context.spaceIdToNamespace(spaceId);
      this.alertingEventLogger.initialize({
        ruleId,
        ruleType: this.ruleType,
        consumer: this.ruleConsumer,
        spaceId,
        executionId: this.executionId,
        taskScheduledAt: this.taskInstance.scheduledAt,
        ...(namespace ? {
          namespace
        } : {})
      });
      this.alertingEventLogger.start(this.runDate);
      if (_elasticApmNode.default.currentTransaction) {
        _elasticApmNode.default.currentTransaction.name = `Execute Alerting Rule`;
        _elasticApmNode.default.currentTransaction.addLabels({
          alerting_rule_space_id: spaceId,
          alerting_rule_id: ruleId
        });
      }
      this.ruleRunning.start(ruleId, this.context.spaceIdToNamespace(spaceId));
      this.logger.debug(`executing rule ${this.ruleType.id}:${ruleId} at ${this.runDate.toISOString()}`);
      if (startedAt) {
        // Capture how long it took for the rule to start running after being claimed
        this.timer.setDuration(_task_runner_timer.TaskRunnerTimerSpan.StartTaskRun, startedAt);
      }
      const ruleData = await (0, _rule_loader.getDecryptedRule)(this.context, ruleId, spaceId);
      const runRuleParams = (0, _rule_loader.validateRuleAndCreateFakeRequest)({
        ruleData,
        paramValidator: this.ruleType.validate.params,
        ruleId,
        spaceId,
        context: this.context,
        ruleTypeRegistry: this.ruleTypeRegistry
      });

      // Update the consumer
      this.ruleConsumer = runRuleParams.rule.consumer;

      // Update the rule name
      this.alertingEventLogger.setRuleName(runRuleParams.rule.name);

      // Set rule monitoring data
      this.ruleMonitoring.setMonitoring(runRuleParams.rule.monitoring);

      // Load the maintenance windows
      this.maintenanceWindows = await (0, _get_maintenance_windows.getMaintenanceWindows)({
        context: this.context,
        fakeRequest: runRuleParams.fakeRequest,
        logger: this.logger,
        ruleTypeId: this.ruleType.id,
        ruleId,
        ruleTypeCategory: this.ruleType.category
      });

      // Set the event log MW Id field the first time with MWs without scoped queries
      this.maintenanceWindowsWithoutScopedQueryIds = (0, _get_maintenance_windows.filterMaintenanceWindowsIds)({
        maintenanceWindows: this.maintenanceWindows,
        withScopedQuery: false
      });
      if (this.maintenanceWindowsWithoutScopedQueryIds.length) {
        this.alertingEventLogger.setMaintenanceWindowIds(this.maintenanceWindowsWithoutScopedQueryIds);
      }
      (async () => {
        try {
          await runRuleParams.rulesClient.clearExpiredSnoozes({
            rule: runRuleParams.rule,
            version: runRuleParams.version
          });
        } catch (e) {
          // Most likely a 409 conflict error, which is ok, we'll try again at the next rule run
          this.logger.debug(`Failed to clear expired snoozes: ${e.message}`);
        }
      })();
      return runRuleParams;
    });
  }
  async processRunResults({
    schedule,
    stateWithMetrics
  }) {
    const {
      executionStatus: execStatus,
      executionMetrics: execMetrics
    } = await this.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.ProcessRuleRun, async () => {
      const {
        params: {
          alertId: ruleId,
          spaceId
        },
        startedAt,
        schedule: taskSchedule
      } = this.taskInstance;
      let nextRun = null;
      if ((0, _result_type.isOk)(schedule)) {
        nextRun = (0, _lib.getNextRun)({
          startDate: startedAt,
          interval: schedule.value.interval
        });
      } else if (taskSchedule) {
        nextRun = (0, _lib.getNextRun)({
          startDate: startedAt,
          interval: taskSchedule.interval
        });
      }
      const namespace = this.context.spaceIdToNamespace(spaceId);

      // Getting executionStatus for backwards compatibility
      const {
        status: executionStatus
      } = (0, _result_type.map)(stateWithMetrics, ruleRunStateWithMetrics => (0, _lib.executionStatusFromState)({
        stateWithMetrics: ruleRunStateWithMetrics,
        lastExecutionDate: this.runDate,
        ruleResultService: this.ruleResult
      }), err => (0, _lib.executionStatusFromError)(err, this.runDate));

      // New consolidated statuses for lastRun
      const {
        lastRun,
        metrics: executionMetrics
      } = (0, _result_type.map)(stateWithMetrics, ruleRunStateWithMetrics => (0, _last_run_status.lastRunFromState)(ruleRunStateWithMetrics, this.ruleResult), err => (0, _lib.lastRunFromError)(err));
      if (_elasticApmNode.default.currentTransaction) {
        if (executionStatus.status === 'ok' || executionStatus.status === 'active') {
          _elasticApmNode.default.currentTransaction.setOutcome('success');
        } else if (executionStatus.status === 'error' || executionStatus.status === 'unknown') {
          _elasticApmNode.default.currentTransaction.setOutcome('failure');
        } else if (lastRun.outcome === 'succeeded') {
          _elasticApmNode.default.currentTransaction.setOutcome('success');
        } else if (lastRun.outcome === 'failed') {
          _elasticApmNode.default.currentTransaction.setOutcome('failure');
        }
      }
      this.logger.debug(`deprecated ruleRunStatus for ${this.ruleType.id}:${ruleId}: ${JSON.stringify(executionStatus)}`);
      this.logger.debug(`ruleRunStatus for ${this.ruleType.id}:${ruleId}: ${JSON.stringify(lastRun)}`);
      if (executionMetrics) {
        this.logger.debug(`ruleRunMetrics for ${this.ruleType.id}:${ruleId}: ${JSON.stringify(executionMetrics)}`);
      }

      // set start and duration based on event log
      const {
        start,
        duration
      } = this.alertingEventLogger.getStartAndDuration();
      if (null != start) {
        executionStatus.lastExecutionDate = start;
      }
      if (null != duration) {
        executionStatus.lastDuration = (0, _server2.nanosToMillis)(duration);
      }

      // if executionStatus indicates an error, fill in fields in
      this.ruleMonitoring.addHistory({
        duration: executionStatus.lastDuration,
        hasError: executionStatus.error != null,
        runDate: this.runDate
      });
      if (!this.cancelled) {
        this.inMemoryMetrics.increment(_monitoring.IN_MEMORY_METRICS.RULE_EXECUTIONS);
        if (lastRun.outcome === 'failed') {
          this.inMemoryMetrics.increment(_monitoring.IN_MEMORY_METRICS.RULE_FAILURES);
        } else if (executionStatus.error) {
          this.inMemoryMetrics.increment(_monitoring.IN_MEMORY_METRICS.RULE_FAILURES);
        }
        this.logger.debug(`Updating rule task for ${this.ruleType.id} rule with id ${ruleId} - ${JSON.stringify(executionStatus)} - ${JSON.stringify(lastRun)}`);
        await this.updateRuleSavedObjectPostRun(ruleId, namespace, {
          executionStatus: (0, _lib.ruleExecutionStatusToRaw)(executionStatus),
          nextRun,
          lastRun: (0, _last_run_status.lastRunToRaw)(lastRun),
          monitoring: this.ruleMonitoring.getMonitoring()
        });
      }
      if (startedAt) {
        // Capture how long it took for the rule to run after being claimed
        this.timer.setDuration(_task_runner_timer.TaskRunnerTimerSpan.TotalRunDuration, startedAt);
      }
      return {
        executionStatus,
        executionMetrics
      };
    });
    this.alertingEventLogger.done({
      status: execStatus,
      metrics: execMetrics,
      timings: this.timer.toJson()
    });
  }
  async run() {
    this.runDate = new Date();
    const {
      params: {
        alertId: ruleId,
        spaceId
      },
      startedAt,
      state: originalState,
      schedule: taskSchedule
    } = this.taskInstance;
    let stateWithMetrics;
    let schedule;
    try {
      const validatedRuleData = await this.prepareToRun();
      stateWithMetrics = (0, _result_type.asOk)(await this.runRule(validatedRuleData));

      // fetch the rule again to ensure we return the correct schedule as it may have
      // changed during the task execution
      const data = await (0, _rule_loader.getDecryptedRule)(this.context, ruleId, spaceId);
      schedule = (0, _result_type.asOk)(data.rawRule.schedule);
    } catch (err) {
      stateWithMetrics = (0, _result_type.asErr)(err);
      schedule = (0, _result_type.asErr)(err);
    }
    await this.processRunResults({
      schedule,
      stateWithMetrics
    });
    const transformRunStateToTaskState = runStateWithMetrics => {
      return {
        ...(0, _lodash.omit)(runStateWithMetrics, ['metrics']),
        previousStartedAt: startedAt === null || startedAt === void 0 ? void 0 : startedAt.toISOString()
      };
    };
    const getTaskRunError = state => {
      if ((0, _result_type.isErr)(state)) {
        return {
          taskRunError: (0, _server.createTaskRunError)(state.error, (0, _task_running.getErrorSource)(state.error))
        };
      }
      const {
        errors: errorsFromLastRun
      } = this.ruleResult.getLastRunResults();
      if (errorsFromLastRun.length > 0) {
        const isUserError = !errorsFromLastRun.some(lastRunError => !lastRunError.userError);
        const lasRunErrorMessages = errorsFromLastRun.map(lastRunError => lastRunError.message).join(',');
        const errorMessage = `Executing Rule ${this.ruleType.id}:${ruleId} has resulted in the following error(s): ${lasRunErrorMessages}`;
        this.logger.error(errorMessage, {
          tags: [this.ruleType.id, ruleId, 'rule-run-failed']
        });
        return {
          taskRunError: (0, _server.createTaskRunError)(new Error(errorMessage), isUserError ? _server.TaskErrorSource.USER : _server.TaskErrorSource.FRAMEWORK)
        };
      }
      return {};
    };
    return {
      state: (0, _result_type.map)(stateWithMetrics, ruleRunStateWithMetrics => transformRunStateToTaskState(ruleRunStateWithMetrics), err => {
        if ((0, _is_alerting_error.isAlertSavedObjectNotFoundError)(err, ruleId)) {
          const message = `Executing Rule ${spaceId}:${this.ruleType.id}:${ruleId} has resulted in Error: ${(0, _errors.getEsErrorMessage)(err)}`;
          this.logger.debug(message);
        } else {
          const error = this.stackTraceLog ? this.stackTraceLog.message : err;
          const stack = this.stackTraceLog ? this.stackTraceLog.stackTrace : err.stack;
          const message = `Executing Rule ${spaceId}:${this.ruleType.id}:${ruleId} has resulted in Error: ${(0, _errors.getEsErrorMessage)(error)} - ${stack !== null && stack !== void 0 ? stack : ''}`;
          this.logger.error(message, {
            tags: [this.ruleType.id, ruleId, 'rule-run-failed'],
            error: {
              stack_trace: stack
            }
          });
        }
        return originalState;
      }),
      schedule: (0, _result_type.resolveErr)(schedule, error => {
        var _taskSchedule$interva;
        if ((0, _is_alerting_error.isAlertSavedObjectNotFoundError)(error, ruleId)) {
          const spaceMessage = spaceId ? `in the "${spaceId}" space ` : '';
          this.logger.warn(`Unable to execute rule "${ruleId}" ${spaceMessage}because ${error.message} - this rule will not be rescheduled. To restart rule execution, try disabling and re-enabling this rule.`);
          (0, _server.throwUnrecoverableError)(error);
        }
        let retryInterval = (_taskSchedule$interva = taskSchedule === null || taskSchedule === void 0 ? void 0 : taskSchedule.interval) !== null && _taskSchedule$interva !== void 0 ? _taskSchedule$interva : FALLBACK_RETRY_INTERVAL;

        // Set retry interval smaller for ES connectivity errors
        if ((0, _is_alerting_error.isEsUnavailableError)(error, ruleId)) {
          retryInterval = (0, _common.parseDuration)(retryInterval) > (0, _common.parseDuration)(CONNECTIVITY_RETRY_INTERVAL) ? CONNECTIVITY_RETRY_INTERVAL : retryInterval;
        }
        return {
          interval: retryInterval
        };
      }),
      monitoring: this.ruleMonitoring.getMonitoring(),
      ...getTaskRunError(stateWithMetrics)
    };
  }
  async cancel() {
    if (this.cancelled) {
      return;
    }
    this.cancelled = true;
    this.ruleTypeRunner.cancelRun();

    // Write event log entry
    const {
      params: {
        alertId: ruleId,
        spaceId,
        consumer
      },
      schedule: taskSchedule,
      startedAt
    } = this.taskInstance;
    const namespace = this.context.spaceIdToNamespace(spaceId);
    if (consumer && !this.ruleConsumer) {
      this.ruleConsumer = consumer;
    }
    this.logger.debug(`Cancelling rule type ${this.ruleType.id} with id ${ruleId} - execution exceeded rule type timeout of ${this.ruleType.ruleTaskTimeout}`);
    this.logger.debug(`Aborting any in-progress ES searches for rule type ${this.ruleType.id} with id ${ruleId}`);
    this.searchAbortController.abort();
    this.alertingEventLogger.logTimeout();
    this.inMemoryMetrics.increment(_monitoring.IN_MEMORY_METRICS.RULE_TIMEOUTS);
    let nextRun = null;
    if (taskSchedule) {
      nextRun = (0, _lib.getNextRun)({
        startDate: startedAt,
        interval: taskSchedule.interval
      });
    }
    const outcomeMsg = [`${this.ruleType.id}:${ruleId}: execution cancelled due to timeout - exceeded rule type timeout of ${this.ruleType.ruleTaskTimeout}`];
    const date = new Date();
    // Update the rule saved object with execution status
    const executionStatus = {
      lastExecutionDate: date,
      status: 'error',
      error: {
        reason: _types.RuleExecutionStatusErrorReasons.Timeout,
        message: outcomeMsg.join(' ')
      }
    };
    this.logger.debug(`Updating rule task for ${this.ruleType.id} rule with id ${ruleId} - execution error due to timeout`);
    const outcome = 'failed';
    await this.updateRuleSavedObjectPostRun(ruleId, namespace, {
      executionStatus: (0, _lib.ruleExecutionStatusToRaw)(executionStatus),
      lastRun: {
        outcome,
        outcomeOrder: _common.RuleLastRunOutcomeOrderMap[outcome],
        warning: _types.RuleExecutionStatusErrorReasons.Timeout,
        outcomeMsg,
        alertsCount: {}
      },
      monitoring: this.ruleMonitoring.getMonitoring(),
      nextRun: nextRun && new Date(nextRun).getTime() > date.getTime() ? nextRun : null
    });
  }
}
exports.TaskRunner = TaskRunner;