"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAlertSummary = getAlertSummary;
var _authorization = require("../../authorization");
var _alert_summary_from_event_log = require("../../lib/alert_summary_from_event_log");
var _parse_duration = require("../../../common/parse_duration");
var _common = require("../common");
var _get = require("./get");
var _saved_objects = require("../../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getAlertSummary(context, {
  id,
  dateStart,
  numberOfExecutions
}) {
  context.logger.debug(`getAlertSummary(): getting alert ${id}`);
  const rule = await (0, _get.get)(context, {
    id,
    includeLegacyId: true
  });
  await context.authorization.ensureAuthorized({
    ruleTypeId: rule.alertTypeId,
    consumer: rule.consumer,
    operation: _authorization.ReadOperations.GetAlertSummary,
    entity: _authorization.AlertingAuthorizationEntity.Rule
  });
  const dateNow = new Date();
  const durationMillis = (0, _parse_duration.parseDuration)(rule.schedule.interval) * (numberOfExecutions !== null && numberOfExecutions !== void 0 ? numberOfExecutions : 60);
  const defaultDateStart = new Date(dateNow.valueOf() - durationMillis);
  const parsedDateStart = (0, _common.parseDate)(dateStart, 'dateStart', defaultDateStart);
  const eventLogClient = await context.getEventLogClient();
  context.logger.debug(`getAlertSummary(): search the event log for rule ${id}`);
  let events;
  let executionEvents;
  try {
    const [queryResults, executionResults] = await Promise.all([eventLogClient.findEventsBySavedObjectIds(_saved_objects.RULE_SAVED_OBJECT_TYPE, [id], {
      page: 1,
      per_page: 10000,
      start: parsedDateStart.toISOString(),
      sort: [{
        sort_field: '@timestamp',
        sort_order: 'desc'
      }],
      end: dateNow.toISOString(),
      // filter out execute-action event logs
      filter: 'NOT event.action: execute-action AND event.provider: alerting'
    }, rule.legacyId !== null ? [rule.legacyId] : undefined), eventLogClient.findEventsBySavedObjectIds(_saved_objects.RULE_SAVED_OBJECT_TYPE, [id], {
      page: 1,
      per_page: numberOfExecutions !== null && numberOfExecutions !== void 0 ? numberOfExecutions : 60,
      filter: 'event.provider: alerting AND event.action:execute',
      sort: [{
        sort_field: '@timestamp',
        sort_order: 'desc'
      }],
      end: dateNow.toISOString()
    }, rule.legacyId !== null ? [rule.legacyId] : undefined)]);
    events = queryResults.data;
    executionEvents = executionResults.data;
  } catch (err) {
    context.logger.debug(`rulesClient.getAlertSummary(): error searching event log for rule ${id}: ${err.message}`);
    events = [];
    executionEvents = [];
  }
  return (0, _alert_summary_from_event_log.alertSummaryFromEventLog)({
    rule,
    events,
    executionEvents,
    dateStart: parsedDateStart.toISOString(),
    dateEnd: dateNow.toISOString()
  });
}