"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createWrappedScopedClusterClientFactory = createWrappedScopedClusterClientFactory;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createWrappedScopedClusterClientFactory(opts) {
  let numSearches = 0;
  let esSearchDurationMs = 0;
  let totalSearchDurationMs = 0;
  function logMetrics(metrics) {
    numSearches++;
    esSearchDurationMs += metrics.esSearchDuration;
    totalSearchDurationMs += metrics.totalSearchDuration;
  }
  const wrappedClient = wrapScopedClusterClient({
    ...opts,
    logMetricsFn: logMetrics
  });
  return {
    client: () => wrappedClient,
    getMetrics: () => {
      return {
        esSearchDurationMs,
        totalSearchDurationMs,
        numSearches
      };
    }
  };
}
function wrapScopedClusterClient(opts) {
  const {
    scopedClusterClient,
    ...rest
  } = opts;
  return {
    asInternalUser: wrapEsClient({
      ...rest,
      esClient: scopedClusterClient.asInternalUser
    }),
    asCurrentUser: wrapEsClient({
      ...rest,
      esClient: scopedClusterClient.asCurrentUser
    })
  };
}
function wrapEsClient(opts) {
  const {
    esClient,
    ...rest
  } = opts;
  const wrappedClient = esClient.child({});

  // Mutating the functions we want to wrap
  wrappedClient.transport.request = getWrappedTransportRequestFn({
    esClient: wrappedClient,
    ...rest
  });
  wrappedClient.search = getWrappedSearchFn({
    esClient: wrappedClient,
    ...rest
  });
  wrappedClient.eql.search = getWrappedEqlSearchFn({
    esClient: wrappedClient,
    ...rest
  });
  return wrappedClient;
}
function getWrappedTransportRequestFn(opts) {
  const originalRequestFn = opts.esClient.transport.request;
  const requestTimeout = opts.requestTimeout;

  // A bunch of overloads to make TypeScript happy

  async function request(params, options) {
    // Wrap ES|QL requests with an abort signal
    if (params.method === 'POST' && params.path === '/_query') {
      try {
        const requestOptions = options !== null && options !== void 0 ? options : {};
        const start = Date.now();
        opts.logger.debug(`executing ES|QL query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(requestOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
        const result = await originalRequestFn.call(opts.esClient.transport, params, {
          ...requestOptions,
          ...(requestTimeout ? {
            requestTimeout
          } : {}),
          signal: opts.abortController.signal
        });
        const end = Date.now();
        const durationMs = end - start;
        opts.logMetricsFn({
          esSearchDuration: 0,
          totalSearchDuration: durationMs
        });
        return result;
      } catch (e) {
        if (opts.abortController.signal.aborted) {
          throw new Error('ES|QL search has been aborted due to cancelled execution');
        }
        throw e;
      }
    }

    // No wrap
    return await originalRequestFn.call(opts.esClient.transport, params, {
      ...options,
      ...(requestTimeout ? {
        requestTimeout
      } : {})
    });
  }
  return request;
}
function getWrappedEqlSearchFn(opts) {
  const originalEqlSearch = opts.esClient.eql.search;
  const requestTimeout = opts.requestTimeout;

  // A bunch of overloads to make TypeScript happy

  async function search(params, options) {
    try {
      var _took;
      const searchOptions = options !== null && options !== void 0 ? options : {};
      const start = Date.now();
      opts.logger.debug(`executing eql query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(searchOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
      const result = await originalEqlSearch.call(opts.esClient, params, {
        ...searchOptions,
        ...(requestTimeout ? {
          requestTimeout
        } : {}),
        signal: opts.abortController.signal
      });
      const end = Date.now();
      const durationMs = end - start;
      let took = 0;
      if (searchOptions.meta) {
        // when meta: true, response is TransportResult<EqlSearchResponse<TEvent>, unknown>
        took = result.body.took;
      } else {
        // when meta: false, response is EqlSearchResponse<TEvent>
        took = result.took;
      }
      opts.logMetricsFn({
        esSearchDuration: (_took = took) !== null && _took !== void 0 ? _took : 0,
        totalSearchDuration: durationMs
      });
      return result;
    } catch (e) {
      if (opts.abortController.signal.aborted) {
        throw new Error('EQL search has been aborted due to cancelled execution');
      }
      throw e;
    }
  }
  return search;
}
function getWrappedSearchFn(opts) {
  const originalSearch = opts.esClient.search;
  const requestTimeout = opts.requestTimeout;

  // A bunch of overloads to make TypeScript happy

  async function search(params, options) {
    try {
      var _took2;
      const searchOptions = options !== null && options !== void 0 ? options : {};
      const start = Date.now();
      opts.logger.debug(`executing query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(searchOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
      const result = await originalSearch.call(opts.esClient, params, {
        ...searchOptions,
        ...(requestTimeout ? {
          requestTimeout
        } : {}),
        signal: opts.abortController.signal
      });
      const end = Date.now();
      const durationMs = end - start;
      let took = 0;
      if (searchOptions.meta) {
        // when meta: true, response is TransportResult<SearchResponse<TDocument, TAggregations>, unknown>
        took = result.body.took;
      } else {
        // when meta: false, response is SearchResponse<TDocument, TAggregations>
        took = result.took;
      }
      opts.logMetricsFn({
        esSearchDuration: (_took2 = took) !== null && _took2 !== void 0 ? _took2 : 0,
        totalSearchDuration: durationMs
      });
      return result;
    } catch (e) {
      if (opts.abortController.signal.aborted) {
        throw new Error('Search has been aborted due to cancelled execution');
      }
      throw e;
    }
  }
  return search;
}