"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.histogramHandlerFactory = void 0;
var _async = require("async");
var _i18n = require("@kbn/i18n");
var _fieldTypes = require("@kbn/field-types");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _constants = require("@kbn/aiops-log-rate-analysis/constants");
var _actions = require("@kbn/aiops-log-rate-analysis/api/actions");
var _get_category_query = require("@kbn/aiops-log-pattern-analysis/get_category_query");
var _get_histogram_query = require("@kbn/aiops-log-rate-analysis/queries/get_histogram_query");
var _constants2 = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const histogramHandlerFactory = ({
  abortSignal,
  client,
  logDebugMessage,
  logger,
  requestBody,
  responseStream,
  stateHandler,
  version
}) => async (fieldValuePairsCount, significantCategories, significantTerms, overallTimeSeries) => {
  var _requestBody$override, _requestBody$override2;
  function pushHistogramDataLoadingState() {
    responseStream.push((0, _actions.updateLoadingStateAction)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.loadingHistogramData', {
        defaultMessage: 'Loading histogram data.'
      })
    }));
  }
  logDebugMessage(`Fetch ${significantTerms.length} field/value histograms.`);

  // time series filtered by fields
  if (significantTerms.length > 0 && overallTimeSeries !== undefined && !((_requestBody$override = requestBody.overrides) !== null && _requestBody$override !== void 0 && _requestBody$override.regroupOnly)) {
    const fieldValueHistogramQueue = (0, _async.queue)(async function (cp) {
      if (stateHandler.shouldStop()) {
        logDebugMessage('shouldStop abort fetching field/value histograms.');
        fieldValueHistogramQueue.kill();
        responseStream.end();
        return;
      }
      if (overallTimeSeries !== undefined) {
        var _overallTimeSeries$da;
        const histogramQuery = (0, _get_histogram_query.getHistogramQuery)(requestBody, [{
          term: {
            [cp.fieldName]: cp.fieldValue
          }
        }]);
        let cpTimeSeries;
        try {
          cpTimeSeries = (await (0, _mlAggUtils.fetchHistogramsForFields)(client, requestBody.index, histogramQuery,
          // fields
          [{
            fieldName: requestBody.timeFieldName,
            type: _fieldTypes.KBN_FIELD_TYPES.DATE,
            interval: overallTimeSeries.interval,
            min: overallTimeSeries.stats[0],
            max: overallTimeSeries.stats[1]
          }],
          // samplerShardSize
          -1, undefined, abortSignal, stateHandler.sampleProbability(), _constants.RANDOM_SAMPLER_SEED))[0];
        } catch (e) {
          logger.error(`Failed to fetch the histogram data for field/value pair "${cp.fieldName}:${cp.fieldValue}", got: \n${e.toString()}`);
          responseStream.pushError(`Failed to fetch the histogram data for field/value pair "${cp.fieldName}:${cp.fieldValue}".`);
          return;
        }
        const histogram = (_overallTimeSeries$da = overallTimeSeries.data.map(o => {
          var _cpTimeSeries$data$fi, _o$key_as_string2;
          const current = (_cpTimeSeries$data$fi = cpTimeSeries.data.find(d1 => d1.key_as_string === o.key_as_string)) !== null && _cpTimeSeries$data$fi !== void 0 ? _cpTimeSeries$data$fi : {
            doc_count: 0
          };
          if (version === '1') {
            var _o$key_as_string;
            return {
              key: o.key,
              key_as_string: (_o$key_as_string = o.key_as_string) !== null && _o$key_as_string !== void 0 ? _o$key_as_string : '',
              doc_count_significant_term: current.doc_count,
              doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
            };
          }
          return {
            key: o.key,
            key_as_string: (_o$key_as_string2 = o.key_as_string) !== null && _o$key_as_string2 !== void 0 ? _o$key_as_string2 : '',
            doc_count_significant_item: current.doc_count,
            doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
          };
        })) !== null && _overallTimeSeries$da !== void 0 ? _overallTimeSeries$da : [];
        const {
          fieldName,
          fieldValue
        } = cp;
        stateHandler.loaded(1 / fieldValuePairsCount * _constants2.PROGRESS_STEP_HISTOGRAMS, false);
        pushHistogramDataLoadingState();
        responseStream.push((0, _actions.addSignificantItemsHistogramAction)([{
          fieldName,
          fieldValue,
          histogram
        }], version));
      }
    }, _constants2.MAX_CONCURRENT_QUERIES);
    fieldValueHistogramQueue.push(significantTerms);
    await fieldValueHistogramQueue.drain();
  }

  // histograms for text field patterns
  if (overallTimeSeries !== undefined && significantCategories.length > 0 && !((_requestBody$override2 = requestBody.overrides) !== null && _requestBody$override2 !== void 0 && _requestBody$override2.regroupOnly)) {
    const significantCategoriesHistogramQueries = significantCategories.map(d => {
      var _histogramQuery$bool;
      const histogramQuery = (0, _get_histogram_query.getHistogramQuery)(requestBody);
      const categoryQuery = (0, _get_category_query.getCategoryQuery)(d.fieldName, [{
        key: `${d.key}`,
        count: d.doc_count,
        examples: [],
        regex: ''
      }]);
      if (Array.isArray((_histogramQuery$bool = histogramQuery.bool) === null || _histogramQuery$bool === void 0 ? void 0 : _histogramQuery$bool.filter)) {
        var _histogramQuery$bool2, _histogramQuery$bool3;
        (_histogramQuery$bool2 = histogramQuery.bool) === null || _histogramQuery$bool2 === void 0 ? void 0 : (_histogramQuery$bool3 = _histogramQuery$bool2.filter) === null || _histogramQuery$bool3 === void 0 ? void 0 : _histogramQuery$bool3.push(categoryQuery);
      }
      return histogramQuery;
    });
    for (const [i, histogramQuery] of significantCategoriesHistogramQueries.entries()) {
      var _overallTimeSeries$da2;
      const cp = significantCategories[i];
      let catTimeSeries;
      try {
        catTimeSeries = (await (0, _mlAggUtils.fetchHistogramsForFields)(client, requestBody.index, histogramQuery,
        // fields
        [{
          fieldName: requestBody.timeFieldName,
          type: _fieldTypes.KBN_FIELD_TYPES.DATE,
          interval: overallTimeSeries.interval,
          min: overallTimeSeries.stats[0],
          max: overallTimeSeries.stats[1]
        }],
        // samplerShardSize
        -1, undefined, abortSignal, stateHandler.sampleProbability(), _constants.RANDOM_SAMPLER_SEED))[0];
      } catch (e) {
        logger.error(`Failed to fetch the histogram data for field/value pair "${cp.fieldName}:${cp.fieldValue}", got: \n${e.toString()}`);
        responseStream.pushError(`Failed to fetch the histogram data for field/value pair "${cp.fieldName}:${cp.fieldValue}".`);
        return;
      }
      const histogram = (_overallTimeSeries$da2 = overallTimeSeries.data.map(o => {
        var _catTimeSeries$data$f, _o$key_as_string4;
        const current = (_catTimeSeries$data$f = catTimeSeries.data.find(d1 => d1.key_as_string === o.key_as_string)) !== null && _catTimeSeries$data$f !== void 0 ? _catTimeSeries$data$f : {
          doc_count: 0
        };
        if (version === '1') {
          var _o$key_as_string3;
          return {
            key: o.key,
            key_as_string: (_o$key_as_string3 = o.key_as_string) !== null && _o$key_as_string3 !== void 0 ? _o$key_as_string3 : '',
            doc_count_significant_term: current.doc_count,
            doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
          };
        }
        return {
          key: o.key,
          key_as_string: (_o$key_as_string4 = o.key_as_string) !== null && _o$key_as_string4 !== void 0 ? _o$key_as_string4 : '',
          doc_count_significant_item: current.doc_count,
          doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
        };
      })) !== null && _overallTimeSeries$da2 !== void 0 ? _overallTimeSeries$da2 : [];
      const {
        fieldName,
        fieldValue
      } = cp;
      stateHandler.loaded(1 / fieldValuePairsCount * _constants2.PROGRESS_STEP_HISTOGRAMS, false);
      pushHistogramDataLoadingState();
      responseStream.push((0, _actions.addSignificantItemsHistogramAction)([{
        fieldName,
        fieldValue,
        histogram
      }], version));
    }
  }
};
exports.histogramHandlerFactory = histogramHandlerFactory;