"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEuiFocusRing = exports.euiOutline = exports.euiFocusRing = void 0;
var _services = require("../../services");
var _functions = require("../functions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * It is best practice to utilize the browser's default `outline` property for handling focus rings.
 * However, some components need to be forced to have the same behavior, or adjust the display.
 * This function re-applies the same default outline with a couple parameters
 * @param euiTheme UseEuiTheme
 * @param offset Accepts a specific measurement or 'inset', 'outset' or 'center' to adjust outline position
 * @param color Accepts any CSS color
 */

var euiOutline = function euiOutline(_ref) {
  var euiTheme = _ref.euiTheme;
  var offset = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'center';
  var color = arguments.length > 2 ? arguments[2] : undefined;
  // Width is enforced as a constant at the global theme layer
  var outlineWidth = euiTheme.focus.width;
  var outlineColor = color || euiTheme.focus.color;
  var outlineOffset = offset;
  if (offset === 'inset') {
    outlineOffset = "-".concat(outlineWidth);
  } else if (offset === 'outset') {
    outlineOffset = "".concat(outlineWidth);
  } else if (offset === 'center') {
    outlineOffset = (0, _functions.mathWithUnits)(outlineWidth, function (x) {
      return x / -2;
    });
  }

  // This is a separate function from `euiFocusRing` because some EUI components
  // need the outline styles only, not the :focus-visible settings (e.g. - :focus-within usage)
  return "\n    outline: ".concat(outlineWidth, " solid ").concat(outlineColor, ";\n    outline-offset: ").concat(outlineOffset, ";\n  ");
};
exports.euiOutline = euiOutline;
var euiFocusRing = function euiFocusRing(euiThemeContext) {
  var offset = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'center';
  var options = arguments.length > 2 ? arguments[2] : undefined;
  // This function utilizes `focus-visible` to turn on focus outlines.
  // But this is browser-dependent:
  // 👉 Safari and Firefox innately respect only showing the outline with keyboard only
  // 💔 But they don't allow coloring of the 'auto'/default outline, so contrast is no good in dark mode.
  // 👉 For these browsers we use the solid type in order to match with `currentColor`.
  // 😦 Which does means the outline will be square
  return "\n    ".concat(euiOutline(euiThemeContext, offset, options === null || options === void 0 ? void 0 : options.color).trim(), "\n\n    // \uD83D\uDC40 Chrome respects :focus-visible and allows coloring the `auto` style\n    &:focus-visible {\n      outline-style: auto;\n    }\n\n    // \uD83D\uDE45\u200D\u2640\uFE0F But Chrome also needs to have the outline forcefully removed from regular `:focus` state\n    &:not(:focus-visible) {\n      outline: none;\n    }\n  ");
};
exports.euiFocusRing = euiFocusRing;
var useEuiFocusRing = function useEuiFocusRing(offset, color) {
  var euiTheme = (0, _services.useEuiTheme)();
  return euiFocusRing(euiTheme, offset, {
    color: color
  });
};
exports.useEuiFocusRing = useEuiFocusRing;