"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Docstore = exports.BaseEntityStore = exports.BaseFileStore = exports.BaseCache = exports.BaseListChatMessageHistory = exports.BaseChatMessageHistory = exports.BasePromptValue = exports.mapStoredMessageToChatMessage = exports.ChatGenerationChunk = exports.ChatMessageChunk = exports.coerceMessageLikeToMessage = exports.isBaseMessageChunk = exports.isBaseMessage = exports.ChatMessage = exports.ToolMessageChunk = exports.ToolMessage = exports.FunctionMessageChunk = exports.FunctionMessage = exports.SystemChatMessage = exports.AIChatMessage = exports.HumanChatMessage = exports.BaseChatMessage = exports.SystemMessageChunk = exports.SystemMessage = exports.AIMessageChunk = exports.AIMessage = exports.HumanMessageChunk = exports.HumanMessage = exports.BaseMessageChunk = exports.BaseMessage = exports.GenerationChunk = exports.RUN_KEY = void 0;
const serializable_js_1 = require("../load/serializable.cjs");
exports.RUN_KEY = "__run";
/**
 * Chunk of a single generation. Used for streaming.
 */
class GenerationChunk {
    constructor(fields) {
        Object.defineProperty(this, "text", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        Object.defineProperty(this, "generationInfo", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.text = fields.text;
        this.generationInfo = fields.generationInfo;
    }
    concat(chunk) {
        return new GenerationChunk({
            text: this.text + chunk.text,
            generationInfo: {
                ...this.generationInfo,
                ...chunk.generationInfo,
            },
        });
    }
}
exports.GenerationChunk = GenerationChunk;
function mergeContent(firstContent, secondContent) {
    // If first content is a string
    if (typeof firstContent === "string") {
        if (typeof secondContent === "string") {
            return firstContent + secondContent;
        }
        else {
            return [{ type: "text", text: firstContent }, ...secondContent];
        }
        // If both are arrays
    }
    else if (Array.isArray(secondContent)) {
        return [...firstContent, ...secondContent];
        // If the first content is a list and second is a string
    }
    else {
        // Otherwise, add the second content as a new element of the list
        return [...firstContent, { type: "text", text: secondContent }];
    }
}
/**
 * Base class for all types of messages in a conversation. It includes
 * properties like `content`, `name`, and `additional_kwargs`. It also
 * includes methods like `toDict()` and `_getType()`.
 */
class BaseMessage extends serializable_js_1.Serializable {
    /**
     * @deprecated
     * Use {@link BaseMessage.content} instead.
     */
    get text() {
        return typeof this.content === "string" ? this.content : "";
    }
    constructor(fields, 
    /** @deprecated */
    kwargs) {
        if (typeof fields === "string") {
            // eslint-disable-next-line no-param-reassign
            fields = { content: fields, additional_kwargs: kwargs };
        }
        // Make sure the default value for additional_kwargs is passed into super() for serialization
        if (!fields.additional_kwargs) {
            // eslint-disable-next-line no-param-reassign
            fields.additional_kwargs = {};
        }
        super(fields);
        Object.defineProperty(this, "lc_namespace", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: ["langchain", "schema"]
        });
        Object.defineProperty(this, "lc_serializable", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        /** The content of the message. */
        Object.defineProperty(this, "content", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /** The name of the message sender in a multi-user chat. */
        Object.defineProperty(this, "name", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /** Additional keyword arguments */
        Object.defineProperty(this, "additional_kwargs", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.name = fields.name;
        this.content = fields.content;
        this.additional_kwargs = fields.additional_kwargs;
    }
    toDict() {
        return {
            type: this._getType(),
            data: this.toJSON()
                .kwargs,
        };
    }
    toChunk() {
        const type = this._getType();
        if (type === "human") {
            // eslint-disable-next-line @typescript-eslint/no-use-before-define
            return new HumanMessageChunk({ ...this });
        }
        else if (type === "ai") {
            // eslint-disable-next-line @typescript-eslint/no-use-before-define
            return new AIMessageChunk({ ...this });
        }
        else if (type === "system") {
            // eslint-disable-next-line @typescript-eslint/no-use-before-define
            return new SystemMessageChunk({ ...this });
        }
        else if (type === "function") {
            // eslint-disable-next-line @typescript-eslint/no-use-before-define
            return new FunctionMessageChunk({ ...this });
            // eslint-disable-next-line @typescript-eslint/no-use-before-define
        }
        else if (ChatMessage.isInstance(this)) {
            // eslint-disable-next-line @typescript-eslint/no-use-before-define
            return new ChatMessageChunk({ ...this });
        }
        else {
            throw new Error("Unknown message type.");
        }
    }
}
exports.BaseMessage = BaseMessage;
function isOpenAIToolCallArray(value) {
    return (Array.isArray(value) &&
        value.every((v) => typeof v.index === "number"));
}
/**
 * Represents a chunk of a message, which can be concatenated with other
 * message chunks. It includes a method `_merge_kwargs_dict()` for merging
 * additional keyword arguments from another `BaseMessageChunk` into this
 * one. It also overrides the `__add__()` method to support concatenation
 * of `BaseMessageChunk` instances.
 */
class BaseMessageChunk extends BaseMessage {
    static _mergeAdditionalKwargs(left, right) {
        const merged = { ...left };
        for (const [key, value] of Object.entries(right)) {
            if (merged[key] === undefined) {
                merged[key] = value;
            }
            else if (typeof merged[key] !== typeof value) {
                throw new Error(`additional_kwargs[${key}] already exists in the message chunk, but with a different type.`);
            }
            else if (typeof merged[key] === "string") {
                merged[key] = merged[key] + value;
            }
            else if (!Array.isArray(merged[key]) &&
                typeof merged[key] === "object") {
                merged[key] = this._mergeAdditionalKwargs(merged[key], value);
            }
            else if (key === "tool_calls" &&
                isOpenAIToolCallArray(merged[key]) &&
                isOpenAIToolCallArray(value)) {
                for (const toolCall of value) {
                    if (merged[key]?.[toolCall.index] !== undefined) {
                        merged[key] = merged[key]?.map((value, i) => {
                            if (i !== toolCall.index) {
                                return value;
                            }
                            return {
                                ...value,
                                ...toolCall,
                                function: {
                                    name: toolCall.function.name ?? value.function.name,
                                    arguments: (value.function.arguments ?? "") +
                                        (toolCall.function.arguments ?? ""),
                                },
                            };
                        });
                    }
                    else {
                        merged[key][toolCall.index] = toolCall;
                    }
                }
            }
            else {
                throw new Error(`additional_kwargs[${key}] already exists in this message chunk.`);
            }
        }
        return merged;
    }
}
exports.BaseMessageChunk = BaseMessageChunk;
/**
 * Represents a human message in a conversation.
 */
class HumanMessage extends BaseMessage {
    static lc_name() {
        return "HumanMessage";
    }
    _getType() {
        return "human";
    }
}
exports.HumanMessage = HumanMessage;
/**
 * Represents a chunk of a human message, which can be concatenated with
 * other human message chunks.
 */
class HumanMessageChunk extends BaseMessageChunk {
    static lc_name() {
        return "HumanMessageChunk";
    }
    _getType() {
        return "human";
    }
    concat(chunk) {
        return new HumanMessageChunk({
            content: mergeContent(this.content, chunk.content),
            additional_kwargs: HumanMessageChunk._mergeAdditionalKwargs(this.additional_kwargs, chunk.additional_kwargs),
        });
    }
}
exports.HumanMessageChunk = HumanMessageChunk;
/**
 * Represents an AI message in a conversation.
 */
class AIMessage extends BaseMessage {
    static lc_name() {
        return "AIMessage";
    }
    _getType() {
        return "ai";
    }
}
exports.AIMessage = AIMessage;
/**
 * Represents a chunk of an AI message, which can be concatenated with
 * other AI message chunks.
 */
class AIMessageChunk extends BaseMessageChunk {
    static lc_name() {
        return "AIMessageChunk";
    }
    _getType() {
        return "ai";
    }
    concat(chunk) {
        return new AIMessageChunk({
            content: mergeContent(this.content, chunk.content),
            additional_kwargs: AIMessageChunk._mergeAdditionalKwargs(this.additional_kwargs, chunk.additional_kwargs),
        });
    }
}
exports.AIMessageChunk = AIMessageChunk;
/**
 * Represents a system message in a conversation.
 */
class SystemMessage extends BaseMessage {
    static lc_name() {
        return "SystemMessage";
    }
    _getType() {
        return "system";
    }
}
exports.SystemMessage = SystemMessage;
/**
 * Represents a chunk of a system message, which can be concatenated with
 * other system message chunks.
 */
class SystemMessageChunk extends BaseMessageChunk {
    static lc_name() {
        return "SystemMessageChunk";
    }
    _getType() {
        return "system";
    }
    concat(chunk) {
        return new SystemMessageChunk({
            content: mergeContent(this.content, chunk.content),
            additional_kwargs: SystemMessageChunk._mergeAdditionalKwargs(this.additional_kwargs, chunk.additional_kwargs),
        });
    }
}
exports.SystemMessageChunk = SystemMessageChunk;
/**
 * @deprecated
 * Use {@link BaseMessage} instead.
 */
exports.BaseChatMessage = BaseMessage;
/**
 * @deprecated
 * Use {@link HumanMessage} instead.
 */
exports.HumanChatMessage = HumanMessage;
/**
 * @deprecated
 * Use {@link AIMessage} instead.
 */
exports.AIChatMessage = AIMessage;
/**
 * @deprecated
 * Use {@link SystemMessage} instead.
 */
exports.SystemChatMessage = SystemMessage;
/**
 * Represents a function message in a conversation.
 */
class FunctionMessage extends BaseMessage {
    static lc_name() {
        return "FunctionMessage";
    }
    constructor(fields, 
    /** @deprecated */
    name) {
        if (typeof fields === "string") {
            // eslint-disable-next-line no-param-reassign, @typescript-eslint/no-non-null-assertion
            fields = { content: fields, name: name };
        }
        super(fields);
    }
    _getType() {
        return "function";
    }
}
exports.FunctionMessage = FunctionMessage;
/**
 * Represents a chunk of a function message, which can be concatenated
 * with other function message chunks.
 */
class FunctionMessageChunk extends BaseMessageChunk {
    static lc_name() {
        return "FunctionMessageChunk";
    }
    _getType() {
        return "function";
    }
    concat(chunk) {
        return new FunctionMessageChunk({
            content: mergeContent(this.content, chunk.content),
            additional_kwargs: FunctionMessageChunk._mergeAdditionalKwargs(this.additional_kwargs, chunk.additional_kwargs),
            name: this.name ?? "",
        });
    }
}
exports.FunctionMessageChunk = FunctionMessageChunk;
/**
 * Represents a tool message in a conversation.
 */
class ToolMessage extends BaseMessage {
    static lc_name() {
        return "ToolMessage";
    }
    constructor(fields, tool_call_id, name) {
        if (typeof fields === "string") {
            // eslint-disable-next-line no-param-reassign, @typescript-eslint/no-non-null-assertion
            fields = { content: fields, name, tool_call_id: tool_call_id };
        }
        super(fields);
        Object.defineProperty(this, "tool_call_id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.tool_call_id = fields.tool_call_id;
    }
    _getType() {
        return "tool";
    }
}
exports.ToolMessage = ToolMessage;
/**
 * Represents a chunk of a function message, which can be concatenated
 * with other function message chunks.
 */
class ToolMessageChunk extends BaseMessageChunk {
    constructor(fields) {
        super(fields);
        Object.defineProperty(this, "tool_call_id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.tool_call_id = fields.tool_call_id;
    }
    static lc_name() {
        return "ToolMessageChunk";
    }
    _getType() {
        return "tool";
    }
    concat(chunk) {
        return new ToolMessageChunk({
            content: mergeContent(this.content, chunk.content),
            additional_kwargs: ToolMessageChunk._mergeAdditionalKwargs(this.additional_kwargs, chunk.additional_kwargs),
            tool_call_id: this.tool_call_id,
        });
    }
}
exports.ToolMessageChunk = ToolMessageChunk;
/**
 * Represents a chat message in a conversation.
 */
class ChatMessage extends BaseMessage {
    static lc_name() {
        return "ChatMessage";
    }
    constructor(fields, role) {
        if (typeof fields === "string") {
            // eslint-disable-next-line no-param-reassign, @typescript-eslint/no-non-null-assertion
            fields = { content: fields, role: role };
        }
        super(fields);
        Object.defineProperty(this, "role", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.role = fields.role;
    }
    _getType() {
        return "generic";
    }
    static isInstance(message) {
        return message._getType() === "generic";
    }
}
exports.ChatMessage = ChatMessage;
function isBaseMessage(messageLike) {
    return typeof messageLike?._getType === "function";
}
exports.isBaseMessage = isBaseMessage;
function isBaseMessageChunk(messageLike) {
    return (isBaseMessage(messageLike) &&
        typeof messageLike.concat === "function");
}
exports.isBaseMessageChunk = isBaseMessageChunk;
function coerceMessageLikeToMessage(messageLike) {
    if (typeof messageLike === "string") {
        return new HumanMessage(messageLike);
    }
    else if (isBaseMessage(messageLike)) {
        return messageLike;
    }
    const [type, content] = messageLike;
    if (type === "human" || type === "user") {
        return new HumanMessage({ content });
    }
    else if (type === "ai" || type === "assistant") {
        return new AIMessage({ content });
    }
    else if (type === "system") {
        return new SystemMessage({ content });
    }
    else {
        throw new Error(`Unable to coerce message from array: only human, AI, or system message coercion is currently supported.`);
    }
}
exports.coerceMessageLikeToMessage = coerceMessageLikeToMessage;
/**
 * Represents a chunk of a chat message, which can be concatenated with
 * other chat message chunks.
 */
class ChatMessageChunk extends BaseMessageChunk {
    static lc_name() {
        return "ChatMessageChunk";
    }
    constructor(fields, role) {
        if (typeof fields === "string") {
            // eslint-disable-next-line no-param-reassign, @typescript-eslint/no-non-null-assertion
            fields = { content: fields, role: role };
        }
        super(fields);
        Object.defineProperty(this, "role", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.role = fields.role;
    }
    _getType() {
        return "generic";
    }
    concat(chunk) {
        return new ChatMessageChunk({
            content: mergeContent(this.content, chunk.content),
            additional_kwargs: ChatMessageChunk._mergeAdditionalKwargs(this.additional_kwargs, chunk.additional_kwargs),
            role: this.role,
        });
    }
}
exports.ChatMessageChunk = ChatMessageChunk;
class ChatGenerationChunk extends GenerationChunk {
    constructor(fields) {
        super(fields);
        Object.defineProperty(this, "message", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.message = fields.message;
    }
    concat(chunk) {
        return new ChatGenerationChunk({
            text: this.text + chunk.text,
            generationInfo: {
                ...this.generationInfo,
                ...chunk.generationInfo,
            },
            message: this.message.concat(chunk.message),
        });
    }
}
exports.ChatGenerationChunk = ChatGenerationChunk;
/**
 * Maps messages from an older format (V1) to the current `StoredMessage`
 * format. If the message is already in the `StoredMessage` format, it is
 * returned as is. Otherwise, it transforms the V1 message into a
 * `StoredMessage`. This function is important for maintaining
 * compatibility with older message formats.
 */
function mapV1MessageToStoredMessage(message) {
    // TODO: Remove this mapper when we deprecate the old message format.
    if (message.data !== undefined) {
        return message;
    }
    else {
        const v1Message = message;
        return {
            type: v1Message.type,
            data: {
                content: v1Message.text,
                role: v1Message.role,
                name: undefined,
                tool_call_id: undefined,
            },
        };
    }
}
function mapStoredMessageToChatMessage(message) {
    const storedMessage = mapV1MessageToStoredMessage(message);
    switch (storedMessage.type) {
        case "human":
            return new HumanMessage(storedMessage.data);
        case "ai":
            return new AIMessage(storedMessage.data);
        case "system":
            return new SystemMessage(storedMessage.data);
        case "function":
            if (storedMessage.data.name === undefined) {
                throw new Error("Name must be defined for function messages");
            }
            return new FunctionMessage(storedMessage.data);
        case "tool":
            if (storedMessage.data.tool_call_id === undefined) {
                throw new Error("Tool call ID must be defined for tool messages");
            }
            return new ToolMessage(storedMessage.data);
        case "chat": {
            if (storedMessage.data.role === undefined) {
                throw new Error("Role must be defined for chat messages");
            }
            return new ChatMessage(storedMessage.data);
        }
        default:
            throw new Error(`Got unexpected type: ${storedMessage.type}`);
    }
}
exports.mapStoredMessageToChatMessage = mapStoredMessageToChatMessage;
/**
 * Base PromptValue class. All prompt values should extend this class.
 */
class BasePromptValue extends serializable_js_1.Serializable {
}
exports.BasePromptValue = BasePromptValue;
/**
 * Base class for all chat message histories. All chat message histories
 * should extend this class.
 */
class BaseChatMessageHistory extends serializable_js_1.Serializable {
}
exports.BaseChatMessageHistory = BaseChatMessageHistory;
/**
 * Base class for all list chat message histories. All list chat message
 * histories should extend this class.
 */
class BaseListChatMessageHistory extends serializable_js_1.Serializable {
    addUserMessage(message) {
        return this.addMessage(new HumanMessage(message));
    }
    addAIChatMessage(message) {
        return this.addMessage(new AIMessage(message));
    }
}
exports.BaseListChatMessageHistory = BaseListChatMessageHistory;
/**
 * Base class for all caches. All caches should extend this class.
 */
class BaseCache {
}
exports.BaseCache = BaseCache;
/**
 * Base class for all file stores. All file stores should extend this
 * class.
 */
class BaseFileStore extends serializable_js_1.Serializable {
}
exports.BaseFileStore = BaseFileStore;
/**
 * Base class for all entity stores. All entity stores should extend this
 * class.
 */
class BaseEntityStore extends serializable_js_1.Serializable {
}
exports.BaseEntityStore = BaseEntityStore;
/**
 * Abstract class for a document store. All document stores should extend
 * this class.
 */
class Docstore {
}
exports.Docstore = Docstore;
