/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { __awaiter, __generator, __read } from "tslib";
import { createInstrumentDescriptorWithView, } from '../InstrumentDescriptor';
import { Meter } from '../Meter';
import { isNotNullish } from '../utils';
import { AsyncMetricStorage } from './AsyncMetricStorage';
import { MetricStorageRegistry } from './MetricStorageRegistry';
import { MultiMetricStorage } from './MultiWritableMetricStorage';
import { ObservableRegistry } from './ObservableRegistry';
import { SyncMetricStorage } from './SyncMetricStorage';
import { AttributesProcessor } from '../view/AttributesProcessor';
/**
 * An internal record for shared meter provider states.
 */
var MeterSharedState = /** @class */ (function () {
    function MeterSharedState(_meterProviderSharedState, _instrumentationScope) {
        this._meterProviderSharedState = _meterProviderSharedState;
        this._instrumentationScope = _instrumentationScope;
        this.metricStorageRegistry = new MetricStorageRegistry();
        this.observableRegistry = new ObservableRegistry();
        this.meter = new Meter(this);
    }
    MeterSharedState.prototype.registerMetricStorage = function (descriptor) {
        var storages = this._registerMetricStorage(descriptor, SyncMetricStorage);
        if (storages.length === 1) {
            return storages[0];
        }
        return new MultiMetricStorage(storages);
    };
    MeterSharedState.prototype.registerAsyncMetricStorage = function (descriptor) {
        var storages = this._registerMetricStorage(descriptor, AsyncMetricStorage);
        return storages;
    };
    /**
     * @param collector opaque handle of {@link MetricCollector} which initiated the collection.
     * @param collectionTime the HrTime at which the collection was initiated.
     * @param options options for collection.
     * @returns the list of metric data collected.
     */
    MeterSharedState.prototype.collect = function (collector, collectionTime, options) {
        return __awaiter(this, void 0, void 0, function () {
            var errors, metricDataList;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.observableRegistry.observe(collectionTime, options === null || options === void 0 ? void 0 : options.timeoutMillis)];
                    case 1:
                        errors = _a.sent();
                        metricDataList = Array.from(this.metricStorageRegistry.getStorages(collector))
                            .map(function (metricStorage) {
                            return metricStorage.collect(collector, _this._meterProviderSharedState.metricCollectors, collectionTime);
                        })
                            .filter(isNotNullish);
                        return [2 /*return*/, {
                                scopeMetrics: {
                                    scope: this._instrumentationScope,
                                    metrics: metricDataList.filter(isNotNullish),
                                },
                                errors: errors,
                            }];
                }
            });
        });
    };
    MeterSharedState.prototype._registerMetricStorage = function (descriptor, MetricStorageType) {
        var _this = this;
        var views = this._meterProviderSharedState.viewRegistry.findViews(descriptor, this._instrumentationScope);
        var storages = views.map(function (view) {
            var viewDescriptor = createInstrumentDescriptorWithView(view, descriptor);
            var compatibleStorage = _this.metricStorageRegistry.findOrUpdateCompatibleStorage(viewDescriptor);
            if (compatibleStorage != null) {
                return compatibleStorage;
            }
            var aggregator = view.aggregation.createAggregator(viewDescriptor);
            var viewStorage = new MetricStorageType(viewDescriptor, aggregator, view.attributesProcessor);
            _this.metricStorageRegistry.register(viewStorage);
            return viewStorage;
        });
        // Fallback to the per-collector aggregations if no view is configured for the instrument.
        if (storages.length === 0) {
            var perCollectorAggregations = this._meterProviderSharedState.selectAggregations(descriptor.type);
            var collectorStorages = perCollectorAggregations.map(function (_a) {
                var _b = __read(_a, 2), collector = _b[0], aggregation = _b[1];
                var compatibleStorage = _this.metricStorageRegistry.findOrUpdateCompatibleCollectorStorage(collector, descriptor);
                if (compatibleStorage != null) {
                    return compatibleStorage;
                }
                var aggregator = aggregation.createAggregator(descriptor);
                var storage = new MetricStorageType(descriptor, aggregator, AttributesProcessor.Noop());
                _this.metricStorageRegistry.registerForCollector(collector, storage);
                return storage;
            });
            storages = storages.concat(collectorStorages);
        }
        return storages;
    };
    return MeterSharedState;
}());
export { MeterSharedState };
//# sourceMappingURL=MeterSharedState.js.map