/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { __awaiter, __extends, __generator, __read, __spreadArray } from "tslib";
import * as api from '@opentelemetry/api';
import { internal, ExportResultCode, globalErrorHandler, unrefTimer, } from '@opentelemetry/core';
import { MetricReader } from './MetricReader';
import { callWithTimeout, TimeoutError } from '../utils';
import { diag } from '@opentelemetry/api';
/**
 * {@link MetricReader} which collects metrics based on a user-configurable time interval, and passes the metrics to
 * the configured {@link PushMetricExporter}
 */
var PeriodicExportingMetricReader = /** @class */ (function (_super) {
    __extends(PeriodicExportingMetricReader, _super);
    function PeriodicExportingMetricReader(options) {
        var _a, _b, _c, _d;
        var _this = _super.call(this, {
            aggregationSelector: (_a = options.exporter.selectAggregation) === null || _a === void 0 ? void 0 : _a.bind(options.exporter),
            aggregationTemporalitySelector: (_b = options.exporter.selectAggregationTemporality) === null || _b === void 0 ? void 0 : _b.bind(options.exporter),
        }) || this;
        if (options.exportIntervalMillis !== undefined &&
            options.exportIntervalMillis <= 0) {
            throw Error('exportIntervalMillis must be greater than 0');
        }
        if (options.exportTimeoutMillis !== undefined &&
            options.exportTimeoutMillis <= 0) {
            throw Error('exportTimeoutMillis must be greater than 0');
        }
        if (options.exportTimeoutMillis !== undefined &&
            options.exportIntervalMillis !== undefined &&
            options.exportIntervalMillis < options.exportTimeoutMillis) {
            throw Error('exportIntervalMillis must be greater than or equal to exportTimeoutMillis');
        }
        _this._exportInterval = (_c = options.exportIntervalMillis) !== null && _c !== void 0 ? _c : 60000;
        _this._exportTimeout = (_d = options.exportTimeoutMillis) !== null && _d !== void 0 ? _d : 30000;
        _this._exporter = options.exporter;
        return _this;
    }
    PeriodicExportingMetricReader.prototype._runOnce = function () {
        return __awaiter(this, void 0, void 0, function () {
            var err_1;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        _a.trys.push([0, 2, , 3]);
                        return [4 /*yield*/, callWithTimeout(this._doRun(), this._exportTimeout)];
                    case 1:
                        _a.sent();
                        return [3 /*break*/, 3];
                    case 2:
                        err_1 = _a.sent();
                        if (err_1 instanceof TimeoutError) {
                            api.diag.error('Export took longer than %s milliseconds and timed out.', this._exportTimeout);
                            return [2 /*return*/];
                        }
                        globalErrorHandler(err_1);
                        return [3 /*break*/, 3];
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    PeriodicExportingMetricReader.prototype._doRun = function () {
        var _a, _b;
        return __awaiter(this, void 0, void 0, function () {
            var _c, resourceMetrics, errors, doExport;
            var _d;
            var _this = this;
            return __generator(this, function (_e) {
                switch (_e.label) {
                    case 0: return [4 /*yield*/, this.collect({
                            timeoutMillis: this._exportTimeout,
                        })];
                    case 1:
                        _c = _e.sent(), resourceMetrics = _c.resourceMetrics, errors = _c.errors;
                        if (errors.length > 0) {
                            (_d = api.diag).error.apply(_d, __spreadArray(['PeriodicExportingMetricReader: metrics collection errors'], __read(errors), false));
                        }
                        doExport = function () { return __awaiter(_this, void 0, void 0, function () {
                            var result;
                            return __generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0: return [4 /*yield*/, internal._export(this._exporter, resourceMetrics)];
                                    case 1:
                                        result = _a.sent();
                                        if (result.code !== ExportResultCode.SUCCESS) {
                                            throw new Error("PeriodicExportingMetricReader: metrics export failed (error " + result.error + ")");
                                        }
                                        return [2 /*return*/];
                                }
                            });
                        }); };
                        if (!resourceMetrics.resource.asyncAttributesPending) return [3 /*break*/, 2];
                        (_b = (_a = resourceMetrics.resource).waitForAsyncAttributes) === null || _b === void 0 ? void 0 : _b.call(_a).then(doExport, function (err) {
                            return diag.debug('Error while resolving async portion of resource: ', err);
                        });
                        return [3 /*break*/, 4];
                    case 2: return [4 /*yield*/, doExport()];
                    case 3:
                        _e.sent();
                        _e.label = 4;
                    case 4: return [2 /*return*/];
                }
            });
        });
    };
    PeriodicExportingMetricReader.prototype.onInitialized = function () {
        var _this = this;
        // start running the interval as soon as this reader is initialized and keep handle for shutdown.
        this._interval = setInterval(function () {
            // this._runOnce never rejects. Using void operator to suppress @typescript-eslint/no-floating-promises.
            void _this._runOnce();
        }, this._exportInterval);
        unrefTimer(this._interval);
    };
    PeriodicExportingMetricReader.prototype.onForceFlush = function () {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this._runOnce()];
                    case 1:
                        _a.sent();
                        return [4 /*yield*/, this._exporter.forceFlush()];
                    case 2:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    PeriodicExportingMetricReader.prototype.onShutdown = function () {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this._interval) {
                            clearInterval(this._interval);
                        }
                        return [4 /*yield*/, this._exporter.shutdown()];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    return PeriodicExportingMetricReader;
}(MetricReader));
export { PeriodicExportingMetricReader };
//# sourceMappingURL=PeriodicExportingMetricReader.js.map