"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StorageProcessor = void 0;
const protocol_js_1 = require("../../../protocol/protocol.js");
const assert_js_1 = require("../../../utils/assert.js");
const log_js_1 = require("../../../utils/log.js");
const NetworkProcessor_js_1 = require("../network/NetworkProcessor.js");
/**
 * Responsible for handling the `storage` domain.
 */
class StorageProcessor {
    #browserCdpClient;
    #browsingContextStorage;
    #logger;
    constructor(browserCdpClient, browsingContextStorage, logger) {
        this.#browsingContextStorage = browsingContextStorage;
        this.#browserCdpClient = browserCdpClient;
        this.#logger = logger;
    }
    async getCookies(params) {
        const partitionKey = this.#expandStoragePartitionSpec(params.partition);
        const cdpResponse = await this.#browserCdpClient.sendCommand('Storage.getCookies', {});
        const filteredBiDiCookies = cdpResponse.cookies
            .filter(
        // CDP's partition key is the source origin.
        (c) => c.partitionKey === undefined ||
            c.partitionKey === partitionKey.sourceOrigin)
            .map((c) => this.#cdpToBiDiCookie(c))
            .filter((c) => this.#matchCookie(c, params.filter));
        return {
            cookies: filteredBiDiCookies,
            partitionKey,
        };
    }
    async setCookie(params) {
        const partitionKey = this.#expandStoragePartitionSpec(params.partition);
        const cdpCookie = this.#bidiToCdpCookie(params, partitionKey);
        try {
            await this.#browserCdpClient.sendCommand('Storage.setCookies', {
                cookies: [cdpCookie],
            });
        }
        catch (e) {
            this.#logger?.(log_js_1.LogType.debugError, e);
            throw new protocol_js_1.UnableToSetCookieException(e.toString());
        }
        return {
            partitionKey,
        };
    }
    #expandStoragePartitionSpecByBrowsingContext(descriptor) {
        const browsingContextId = descriptor.context;
        const browsingContext = this.#browsingContextStorage.getContext(browsingContextId);
        const url = NetworkProcessor_js_1.NetworkProcessor.parseUrlString(browsingContext?.url ?? '');
        // Cookie origin should not contain the port.
        // Origin `null` is a special case for local pages.
        const sourceOrigin = url.origin === 'null' ? url.origin : `${url.protocol}//${url.hostname}`;
        return {
            sourceOrigin,
        };
    }
    #expandStoragePartitionSpecByStorageKey(descriptor) {
        let sourceOrigin = undefined;
        if (descriptor.sourceOrigin !== undefined) {
            sourceOrigin = descriptor.sourceOrigin;
        }
        if (sourceOrigin === undefined) {
            throw new protocol_js_1.UnderspecifiedStoragePartitionException('"sourceOrigin" should be set');
        }
        const unsupportedPartitionKeys = new Map();
        // Partition spec is a storage partition.
        // Let partition key be partition spec.
        for (const [key, value] of Object.entries(descriptor)) {
            if (key !== undefined &&
                value !== undefined &&
                !['type', 'sourceOrigin'].includes(key)) {
                unsupportedPartitionKeys.set(key, value);
            }
        }
        if (unsupportedPartitionKeys.size > 0) {
            this.#logger?.(log_js_1.LogType.debugInfo, `Unsupported partition keys: ${JSON.stringify(Object.fromEntries(unsupportedPartitionKeys))}`);
        }
        return {
            sourceOrigin,
        };
    }
    #expandStoragePartitionSpec(partitionSpec) {
        if (partitionSpec === undefined) {
            throw new protocol_js_1.UnderspecifiedStoragePartitionException('partition should be set');
        }
        if (partitionSpec.type === 'context') {
            return this.#expandStoragePartitionSpecByBrowsingContext(partitionSpec);
        }
        (0, assert_js_1.assert)(partitionSpec.type === 'storageKey', 'Unknown partition type');
        return this.#expandStoragePartitionSpecByStorageKey(partitionSpec);
    }
    #bidiToCdpCookie(params, partitionKey) {
        if (params.cookie.value.type !== 'string') {
            // CDP supports only string values in cookies.
            throw new protocol_js_1.UnsupportedOperationException('Only string cookie values are supported');
        }
        const deserializedValue = params.cookie.value.value;
        return {
            name: params.cookie.name,
            value: deserializedValue,
            domain: params.cookie.domain,
            path: params.cookie.path ?? '/',
            secure: params.cookie.secure ?? false,
            httpOnly: params.cookie.httpOnly ?? false,
            // CDP's `partitionKey` is the BiDi's `partition.sourceOrigin`.
            partitionKey: partitionKey.sourceOrigin,
            ...(params.cookie.expiry !== undefined && {
                expires: params.cookie.expiry,
            }),
            ...(params.cookie.sameSite !== undefined && {
                sameSite: StorageProcessor.#sameSiteBiDiToCdp(params.cookie.sameSite),
            }),
            // TODO: extend with CDP-specific properties with `goog:` prefix after
            //  https://github.com/w3c/webdriver-bidi/pull/637
            //  * session: boolean;
            //  * priority: CookiePriority;
            //  * sameParty: boolean;
            //  * sourceScheme: CookieSourceScheme;
            //  * sourcePort: integer;
            //  * partitionKey?: string;
            //  * partitionKeyOpaque?: boolean;
        };
    }
    #cdpToBiDiCookie(cookie) {
        return {
            name: cookie.name,
            value: { type: 'string', value: cookie.value },
            domain: cookie.domain,
            path: cookie.path,
            size: cookie.size,
            httpOnly: cookie.httpOnly,
            secure: cookie.secure,
            sameSite: cookie.sameSite === undefined
                ? "none" /* Network.SameSite.None */
                : StorageProcessor.#sameSiteCdpToBiDi(cookie.sameSite),
            ...(cookie.expires >= 0 ? { expiry: cookie.expires } : undefined),
            // TODO: extend with CDP-specific properties with `goog:` prefix after
            //  https://github.com/w3c/webdriver-bidi/pull/637
            //  * session: boolean;
            //  * priority: CookiePriority;
            //  * sameParty: boolean;
            //  * sourceScheme: CookieSourceScheme;
            //  * sourcePort: integer;
            //  * partitionKey?: string;
            //  * partitionKeyOpaque?: boolean;
        };
    }
    static #sameSiteCdpToBiDi(sameSite) {
        switch (sameSite) {
            case 'Strict':
                return "strict" /* Network.SameSite.Strict */;
            case 'None':
                return "none" /* Network.SameSite.None */;
            case 'Lax':
                return "lax" /* Network.SameSite.Lax */;
            default:
                // Defaults to `Lax`:
                // https://web.dev/articles/samesite-cookies-explained#samesitelax_by_default
                return "lax" /* Network.SameSite.Lax */;
        }
    }
    static #sameSiteBiDiToCdp(sameSite) {
        switch (sameSite) {
            case "strict" /* Network.SameSite.Strict */:
                return 'Strict';
            case "lax" /* Network.SameSite.Lax */:
                return 'Lax';
            case "none" /* Network.SameSite.None */:
                return 'None';
        }
        throw new protocol_js_1.InvalidArgumentException(`Unknown 'sameSite' value ${sameSite}`);
    }
    #matchCookie(cookie, filter) {
        if (filter === undefined) {
            return true;
        }
        return ((filter.domain === undefined || filter.domain === cookie.domain) &&
            (filter.name === undefined || filter.name === cookie.name) &&
            // `value` contains fields `type` and `value`.
            (filter.value === undefined ||
                (filter.value.type === cookie.value.type &&
                    filter.value.value === cookie.value.value)) &&
            (filter.path === undefined || filter.path === cookie.path) &&
            (filter.size === undefined || filter.size === cookie.size) &&
            (filter.httpOnly === undefined || filter.httpOnly === cookie.httpOnly) &&
            (filter.secure === undefined || filter.secure === cookie.secure) &&
            (filter.sameSite === undefined || filter.sameSite === cookie.sameSite) &&
            (filter.expiry === undefined || filter.expiry === cookie.expiry));
    }
}
exports.StorageProcessor = StorageProcessor;
//# sourceMappingURL=StorageProcessor.js.map