"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getColumnsFromVis = void 0;
var _metrics = require("../../common/convert_to_lens/lib/metrics");
var _utils = require("../../common/convert_to_lens/lib/utils");
var _vis_schemas = require("../vis_schemas");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const areVisSchemasValid = (visSchemas, unsupported) => {
  const usedUnsupportedSchemas = unsupported.filter(schema => {
    var _visSchemas$schema;
    return visSchemas[schema] && ((_visSchemas$schema = visSchemas[schema]) === null || _visSchemas$schema === void 0 ? void 0 : _visSchemas$schema.length);
  });
  return !usedUnsupportedSchemas.length;
};
const createLayer = (visType, visSchemas, allMetrics, metricsForLayer, customBucketsWithMetricIds, dataView, {
  splits = [],
  buckets = []
} = {}, percentageModeConfig, dropEmptyRowsInDateHistogram) => {
  const metricColumns = metricsForLayer.flatMap(m => (0, _metrics.convertMetricToColumns)({
    agg: m,
    dataView,
    aggs: allMetrics,
    visType
  }, percentageModeConfig));
  if (metricColumns.includes(null)) {
    return null;
  }
  const metricColumnsWithoutNull = metricColumns;
  const {
    customBucketColumns,
    customBucketsMap
  } = (0, _utils2.getCustomBucketColumns)(visType, customBucketsWithMetricIds, metricColumnsWithoutNull, dataView, allMetrics, dropEmptyRowsInDateHistogram);
  if (customBucketColumns.includes(null)) {
    return null;
  }
  const bucketColumns = (0, _utils2.getBucketColumns)(visType, visSchemas, buckets, dataView, false, metricColumns, dropEmptyRowsInDateHistogram);
  if (!bucketColumns) {
    return null;
  }
  const splitBucketColumns = (0, _utils2.getBucketColumns)(visType, visSchemas, splits, dataView, true, metricColumns, dropEmptyRowsInDateHistogram);
  if (!splitBucketColumns) {
    return null;
  }
  const columns = (0, _utils2.sortColumns)([...metricColumnsWithoutNull, ...bucketColumns, ...splitBucketColumns, ...customBucketColumns], visSchemas, [...buckets, ...splits], metricsForLayer);
  const columnsWithoutReferenced = (0, _utils2.getColumnsWithoutReferenced)(columns);
  return {
    metrics: (0, _utils2.getColumnIds)(columnsWithoutReferenced.filter(с => !с.isBucketed)),
    buckets: {
      all: (0, _utils2.getColumnIds)(columnsWithoutReferenced.filter(c => c.isBucketed)),
      customBuckets: customBucketsMap
    },
    bucketCollapseFn: (0, _utils2.getBucketCollapseFn)(visSchemas.metric, customBucketColumns, customBucketsMap, metricColumnsWithoutNull),
    columnsWithoutReferenced,
    columns
  };
};
const getColumnsFromVis = (vis, timefilter, dataView, {
  splits = [],
  buckets = [],
  unsupported = []
} = {}, config, series) => {
  const {
    dropEmptyRowsInDateHistogram,
    supportMixedSiblingPipelineAggs,
    ...percentageModeConfig
  } = config !== null && config !== void 0 ? config : {
    isPercentageMode: false
  };
  const visSchemas = (0, _vis_schemas.getVisSchemas)(vis, {
    timefilter,
    timeRange: timefilter.getAbsoluteTime()
  });
  if (!(0, _utils2.isValidVis)(visSchemas, supportMixedSiblingPipelineAggs) || !areVisSchemasValid(visSchemas, unsupported)) {
    return null;
  }
  const customBucketsWithMetricIds = (0, _utils.getCustomBucketsFromSiblingAggs)(visSchemas.metric);

  // doesn't support sibbling pipeline aggs with different bucket aggs
  if (!supportMixedSiblingPipelineAggs && customBucketsWithMetricIds.length > 1) {
    return null;
  }
  const metricsWithoutDuplicates = (0, _utils2.getMetricsWithoutDuplicates)(visSchemas.metric);
  const aggs = metricsWithoutDuplicates;
  const layers = [];
  if (series && series.length) {
    for (const {
      metrics: metricAggIds
    } of series) {
      const metricAggIdsLookup = new Set(metricAggIds);
      const metrics = aggs.filter(agg => agg.aggId && metricAggIdsLookup.has((0, _utils.getAggIdAndValue)(agg.aggId)[0]));
      const customBucketsForLayer = customBucketsWithMetricIds.filter(c => c.metricIds.some(m => metricAggIdsLookup.has(m)));
      const layer = createLayer(vis.type.name, visSchemas, aggs, metrics, customBucketsForLayer, dataView, {
        splits,
        buckets
      }, percentageModeConfig, dropEmptyRowsInDateHistogram);
      if (!layer) {
        return null;
      }
      layers.push(layer);
    }
  } else {
    const layer = createLayer(vis.type.name, visSchemas, aggs, aggs, customBucketsWithMetricIds, dataView, {
      splits,
      buckets
    }, percentageModeConfig, dropEmptyRowsInDateHistogram);
    if (!layer) {
      return null;
    }
    layers.push(layer);
  }
  return layers;
};
exports.getColumnsFromVis = getColumnsFromVis;